package com.bxm.game.common.core.scene;

import com.bxm.game.common.core.ErrCode;
import com.bxm.game.common.core.GameException;
import com.bxm.game.common.core.prop.Prop;
import com.bxm.game.common.core.scene.event.AcquiredPropEvent;
import com.bxm.warcar.id.IdGenerator;
import com.bxm.warcar.integration.eventbus.SyncEventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Map;
import java.util.Objects;

/**
 * 抽象的场景服务
 *
 * @author allen
 * @date 2020-10-16
 * @since 1.0
 */
@Slf4j
public abstract class AbstractSceneService<R extends SceneRequest, T extends SceneResponse> implements SceneService<R, T> {

    @Autowired
    private IdGenerator id;
    @Autowired
    private SyncEventPark syncEventPark;

    /**
     * @return 是否需要检查 {@code request} 对象
     */
    protected boolean isRequiredRequest() {
        return false;
    }

    /**
     * 在 {@link #take(SceneRequest)} 方法执行前置检查是否终止。
     * @param request 请求
     * @param attach 附件，在内部可以对其进行设置，在创建响应方法里再使用
     * @return 是否终止，如果终止那么就不会继续执行。接着执行 {@link #afterAbort(SceneRequest)}}
     */
    protected boolean isAbort(R request, Map<Object, Object> attach) {
        return false;
    }

    /**
     * 获得一个道具
     * @param request 请求
     * @param attach 附件，在内部可以对其进行设置，在创建响应方法里再使用
     * @return 道具
     */
    protected abstract Prop takeProp(R request, Map<Object, Object> attach);

    /**
     * 创建响应
     * @param request 请求
     * @param id ID
     * @param prop 获得的道具
     * @param attach 附件
     * @return 响应
     */
    protected abstract T createResponse(R request, String id, Prop prop, Map<Object, Object> attach);

    /**
     * 在中断之后请求
     * @param request 请求对象
     */
    protected void afterAbort(R request) {}

    protected void afterTake(R request, Prop prop) {}

    /**
     * 在 {@link #take(SceneRequest)} 全部执行完成后调用这个方法。
     * @param request 请求
     * @param response 响应
     * @param prop 获得的道具
     * @param attach 附件
     */
    protected void afterCompletion(R request, T response, Prop prop, Map<Object, Object> attach) {}

    @Override
    public T take(R request) {
        if (isRequiredRequest() && Objects.isNull(request)) {
            throw new GameException(ErrCode.BAD_REQUEST);
        }
        Map<Object, Object> attach = Maps.newHashMap();
        if (isAbort(request, attach)) {
            this.afterAbort(request);
            return null;
        }
        Prop prop = takeProp(request, attach);
        if (Objects.isNull(prop)) {
            return null;
        }

        this.afterTake(request, prop);

        String id = this.id.next();
        syncEventPark.post(new AcquiredPropEvent(this, id, getSceneType(), prop, isMultipleGrant()));
        T response = createResponse(request, id, prop, attach);
        if (Objects.isNull(response)) {
            log.error("{} createResponse is null", getClass());
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }

        response.setId(id);
        response.setAssetType(prop.getAssetType());
        response.setPropNum(prop.propNum());
        response.setAcquired(prop.acquired());
        response.setMultipleNum(prop.multipleNum());
        response.setStartAcquireTime(prop.startAcquireTime());
        response.setEndAcquireTime(prop.endAcquireTime());

        this.afterCompletion(request, response, prop, attach);

        return response;
    }

    @Override
    public R convert(String message) {
        if (StringUtils.isBlank(message)) {
            return null;
        }
        return JsonHelper.convert(message, getRequestClass());
    }

    protected SyncEventPark getSyncEventPark() {
        return syncEventPark;
    }
}
