package com.bxm.game.common.core.scene.acquire;

import com.bxm.game.common.core.AppContext;
import com.bxm.game.common.core.prop.AcquiredPropLog;
import com.bxm.game.common.core.prop.Prop;
import com.bxm.game.common.core.prop.PropService;
import com.bxm.game.common.core.scene.AbstractSceneService;
import com.bxm.game.common.core.scene.AcquiringOrMultiplyPropEvent;
import com.bxm.game.common.core.scene.SceneType;
import org.apache.commons.lang.StringUtils;

import java.util.Map;
import java.util.Objects;

/**
 * 领取之前获取的道具。
 * 当之前获取到的道具 {@link Prop#acquired()} 是非立即领取的。
 * 那么需要将之前获取的道具单号通过这个服务来领取。
 * @author allen
 * @date 2020-11-04
 * @since 1.0
 */
public class AcquireSceneServiceImpl extends AbstractSceneService<AcquireSceneRequest, AcquireSceneResponse> {

    private final PropService propService;

    public AcquireSceneServiceImpl(PropService propService) {
        this.propService = propService;
    }

    @Override
    protected Prop takeProp(AcquireSceneRequest request, Map<Object, Object> attach) {
        if (Objects.isNull(request)) {
            return null;
        }
        String appId = AppContext.get().getAppId();
        String uid = AppContext.get().getUid();
        String id = request.getId();
        AcquiredPropLog takeLog = propService.getTakeLog(id);
        if (Objects.isNull(takeLog)) {
            return null;
        }
        if (!(StringUtils.equals(appId, takeLog.getAppId())
                && StringUtils.equals(uid, takeLog.getUid())
                && !takeLog.isAcquired()
                && takeLog.isBetweenAcquireTime())) {
            return null;
        }
        Prop prop = propService.rebuild(takeLog);
        if (Objects.nonNull(prop)) {
            propService.saveTakeLog(takeLog.setAcquired(true));
        }
        return prop;
    }

    @Override
    protected void afterCompletion(AcquireSceneRequest request, AcquireSceneResponse response, Prop prop, Map<Object, Object> attach) {
        getSyncEventPark().post(new AcquiredSceneEvent(this, request, response));
        getSyncEventPark().post(new AcquiringOrMultiplyPropEvent(this, request, response));
    }

    @Override
    protected AcquireSceneResponse createResponse(AcquireSceneRequest request, String id, Prop prop, Map<Object, Object> attach) {
        return new AcquireSceneResponse();
    }

    @Override
    public String getSceneType() {
        return SceneType.ACQUIRE;
    }

    @Override
    public Class<AcquireSceneRequest> getRequestClass() {
        return AcquireSceneRequest.class;
    }
}
