package com.bxm.game.common.core.web;

import com.bxm.game.common.core.ErrCode;
import com.bxm.game.common.core.GameException;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * @author allen
 */
@ControllerAdvice
public class GlobalExceptionHandler {

    @ExceptionHandler({GameException.class})
    public ResponseEntity<Integer> fruitsException(GameException e) {
        ErrCode errCode = e.getErrCode();
        if (Objects.isNull(errCode)) {
            return ResponseEntity.badRequest().body(-1);
        }
        return ResponseEntity.status(errCode.getStatusCode()).body(errCode.getCode());
    }

    @ExceptionHandler({UnsupportedOperationException.class})
    public ResponseEntity<String> unsupportedOperation() {
        return ResponseEntity.notFound().build();
    }

    @ExceptionHandler({HttpMessageNotReadableException.class})
    public ResponseEntity<String> httpMessageNotReadable() {
        return ResponseEntity.badRequest().build();
    }

    @ExceptionHandler({IllegalArgumentException.class, DateTimeParseException.class, MissingServletRequestParameterException.class})
    public ResponseEntity<String> illegalArgument(Throwable throwable) {
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(throwable.getMessage());
    }

    @ExceptionHandler({BindException.class})
    public ResponseEntity<String> bindException(BindException throwable) {
        StringBuilder msg = new StringBuilder();
        List<ObjectError> errors = throwable.getAllErrors();
        for (ObjectError error : errors) {
            msg.append(error.getDefaultMessage());
        }
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(msg.toString());
    }

    @ExceptionHandler({MethodArgumentNotValidException.class})
    public ResponseEntity<String> methodArgumentNotValidException(MethodArgumentNotValidException throwable) {
        StringBuilder msg = new StringBuilder();
        BindingResult bindingResult = throwable.getBindingResult();
        List<ObjectError> errors = bindingResult.getAllErrors();
        for (ObjectError error : errors) {
            msg.append(error.getDefaultMessage());
        }
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(msg.toString());
    }

    @ExceptionHandler(ConstraintViolationException.class)
    public ResponseEntity<String> constraintViolationException(ConstraintViolationException throwable) {
        StringBuilder msg = new StringBuilder();
        Set<ConstraintViolation<?>> constraintViolations = throwable.getConstraintViolations();
        for (ConstraintViolation<?> constraintViolation : constraintViolations) {
            msg.append(constraintViolation.getMessage());
        }
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(msg.toString());
    }
}
