package com.bxm.game.common.core.controllers;

import com.bxm.game.common.core.Key;
import com.bxm.game.common.core.archives.CustomArchiveHandler;
import com.bxm.game.common.core.archives.CustomArchiveHandlerFactory;
import com.bxm.game.common.core.assets.AssetService;
import com.bxm.game.common.core.user.PermanentAtomicService;
import com.bxm.game.common.core.user.TodayAtomicService;
import com.bxm.game.common.core.vo.*;
import com.bxm.warcar.cache.Counter;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;
import java.util.Optional;

/**
 * @author allen
 * @date 2020-11-30
 * @since 1.0
 */
@Slf4j
@RestController
@RequestMapping("/archive")
public class ArchiveController {

    private final CustomArchiveFetcherFactory customArchiveFetcherFactory;
    private final CustomArchiveHandlerFactory customArchiveHandlerFactory;
    private final DataTransfer dataTransfer;
    private final TodayAtomicService todayAtomicService;
    private final PermanentAtomicService permanentAtomicService;
    private final AssetService assetService;
    private final Counter counter;
    private final Key key;

    public ArchiveController(CustomArchiveFetcherFactory customArchiveFetcherFactory,
                             CustomArchiveHandlerFactory customArchiveHandlerFactory, DataTransfer dataTransfer, TodayAtomicService todayAtomicService,
                             PermanentAtomicService permanentAtomicService,
                             AssetService assetService,
                             Counter counter,
                             Key key) {
        this.customArchiveFetcherFactory = customArchiveFetcherFactory;
        this.customArchiveHandlerFactory = customArchiveHandlerFactory;
        this.dataTransfer = dataTransfer;
        this.todayAtomicService = todayAtomicService;
        this.permanentAtomicService = permanentAtomicService;
        this.assetService = assetService;
        this.counter = counter;
        this.key = key;
    }

    @GetMapping("/fetch")
    public ResponseEntity<ArchiveVo> fetch() {
        Map<String, Object> custom = Maps.newHashMap();
        ImmutableSet<CustomArchiveFetcher> fetchers = customArchiveFetcherFactory.getFetchers();
        for (CustomArchiveFetcher fetcher : fetchers) {
            String key = fetcher.getKey();
            if (custom.containsKey(key)) {
                log.warn("duplicated key: {}", key);
            }
            custom.put(key, fetcher.getValue());
        }

        ImmutableSet<CustomArchiveHandler> handlers = customArchiveHandlerFactory.getHandlers();
        for (CustomArchiveHandler handler : handlers) {
            String displayFieldName = handler.getDisplayFieldName();
            if (custom.containsKey(displayFieldName)) {
                log.warn("duplicated key: {}", displayFieldName);
            }
            custom.put(displayFieldName, handler.getArchiveObjects());
        }

        ArchiveVo.ArchiveVoBuilder builder = ArchiveVo.builder();
        ArchiveVo archiveVo = builder
                .today(dataTransfer.transfer(todayAtomicService.getAll()))
                .permanent(dataTransfer.transfer(permanentAtomicService.getAll()))
                .assets(assetService.getAll())
                .custom(custom)
                .continuousSignInDays(Optional.ofNullable(counter.get(key.getSignin())).orElse(0L))
                .time(System.currentTimeMillis())
                .build();

        return ResponseEntity.ok(archiveVo);
    }
}
