package com.bxm.game.common.core.scene.multiple;

import com.bxm.game.common.core.AppContext;
import com.bxm.game.common.core.ErrCode;
import com.bxm.game.common.core.GameException;
import com.bxm.game.common.core.prop.AcquiredPropLog;
import com.bxm.game.common.core.prop.Prop;
import com.bxm.game.common.core.prop.PropService;
import com.bxm.game.common.core.scene.AbstractSceneService;
import com.bxm.game.common.core.scene.AcquiringOrMultiplyPropEvent;
import com.bxm.game.common.core.scene.SceneType;
import com.bxm.game.common.core.user.TemporaryAtomicService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.util.Map;
import java.util.Objects;

/**
 * 翻倍场景
 *
 * @author allen
 * @date 2020-11-26
 * @since 1.0
 */
@Slf4j
public class MultipleSceneServiceImpl extends AbstractSceneService<MultipleSceneRequest, MultipleSceneResponse> implements BeanPostProcessor {

    private final Map<String, MultipleConditional> condtionalMap = Maps.newHashMap();
    private final PropService propService;
    private final TemporaryAtomicService temporaryAtomicService;

    public MultipleSceneServiceImpl(PropService propService, TemporaryAtomicService temporaryAtomicService) {
        this.propService = propService;
        this.temporaryAtomicService = temporaryAtomicService;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (bean instanceof MultipleConditional) {
            MultipleConditional onSceneType = (MultipleConditional) bean;
            String conditionSceneType = onSceneType.getSceneType();
            condtionalMap.put(conditionSceneType, onSceneType);
            log.info("Registered MultipleConditional bean: {}", bean);
        }
        return bean;
    }

    @Override
    public boolean isMultipleGrant() {
        return true;
    }

    @Override
    protected String createId(MultipleSceneRequest request) {
        return request.getId();
    }

    @Override
    protected Prop takeProp(MultipleSceneRequest request, Map<Object, Object> attach) {
        if (Objects.isNull(request)) {
            return null;
        }
        String appId = AppContext.get().getAppId();
        String uid = AppContext.get().getUid();
        String id = request.getId();
        String barrier = getSceneType() + id;
        if (temporaryAtomicService.incrementAndAbove(barrier, 1)) {
            throw new GameException(ErrCode.NOT_MEET_ACQUIRING_CONDITION);
        }
        AcquiredPropLog takeLog = propService.getTakeLog(id);
        if (Objects.isNull(takeLog)) {
            return null;
        }
        if (!(StringUtils.equals(appId, takeLog.getAppId())
                && StringUtils.equals(uid, takeLog.getUid())
                && !takeLog.isAcquired()
                && !takeLog.isCompletedMultiple()
                && takeLog.getMultipleNum() > 1)) {
            return null;
        }
        if (condtionalMap.containsKey(takeLog.getSceneType())
                && !condtionalMap.get(takeLog.getSceneType()).checked(takeLog)) {
            // rollback barrier times
            temporaryAtomicService.incrementBy(barrier, -1);
            throw new GameException(ErrCode.NOT_MEET_MULTIPLE_CONDITION);
        }
        Prop prop = propService.rebuild(takeLog);
        if (Objects.nonNull(prop)) {
            propService.saveTakeLog(takeLog.setAcquired(true).setCompletedMultiple(true));
        }
        return prop;
    }

    @Override
    protected MultipleSceneResponse createResponse(MultipleSceneRequest request, String id, Prop prop, Map<Object, Object> attach) {
        return new MultipleSceneResponse();
    }

    @Override
    protected void afterCompletion(MultipleSceneRequest request, MultipleSceneResponse response, Prop prop, Map<Object, Object> attach) {
        getSyncEventPark().post(new MultipleSceneEvent(this, request, response));
        getSyncEventPark().post(new AcquiringOrMultiplyPropEvent(this, request, response));
    }

    @Override
    public String getSceneType() {
        return SceneType.MULTIPLE;
    }

    @Override
    public Class<MultipleSceneRequest> getRequestClass() {
        return MultipleSceneRequest.class;
    }
}
