package com.bxm.game.common.core.user;

import java.util.Map;
import java.util.function.Predicate;

/**
 * @author allen
 * @date 2020-11-27
 * @since 1.0
 */
public interface AtomicService {

    /**
     * 获取所有的存储数据
     * @return 数据
     */
    Map<String, Object> getAll();

    /**
     * 获取指定 {@code key} 的数值
     * @param key key
     * @return 数值
     */
    long get(String key);

    /**
     * 在当前 {@code key} 对应的数值 +1
     * @param key key
     * @return 增加后的数值
     */
    default long increment(String key) {
        return incrementBy(key, 1);
    }

    /**
     * 在当前 {@code key} 对应的数值增量 {@code value}
     * @param key key
     * @param value value
     * @return 增加后的数值
     */
    long incrementBy(String key, long value);

    /**
     * 在当前 {@code key} 对应的数值 +1，并判断返回的值是否大于 {@code above}
     *
     * @param key key
     * @param above 比较值
     * @return 判断结果
     */
    default boolean incrementAndAbove(String key, long above) {
        return increment(key, (after) -> after > above);
    }

    /**
     * 在当前 {@code key} 对应的数值 +1，并返回 {@code predicate} 判断后的结果
     * @param key key
     * @param predicate 判断接口，参数值为增加后的数值
     * @return 判断结果
     */
    default boolean increment(String key, Predicate<Long> predicate) {
        return incrementBy(key, 1, predicate);
    }

    /**
     * 在当前 {@code key} 对应的数值增量 {@code value}，并返回 {@code predicate} 判断后的结果
     * @param key key
     * @param value value
     * @param predicate 判断接口，参数值为增加后的数值
     * @return 判断结果
     */
    boolean incrementBy(String key, long value, Predicate<Long> predicate);
}
