package com.bxm.game.common.core;

import java.util.Arrays;
import java.util.List;

import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;

/**
 * @author allen
 * @date 2020-10-16
 * @since 1.0
 */
public final class Key {

    public static final int EXPIRE_TIME_TEN_MINUTES = 10 * 60;
    public static final int EXPIRE_TIME_ONE_DAY = 24 * 60 * 60;
    private final AppConfig appConfig;

    public Key(AppConfig appConfig) {
        this.appConfig = appConfig;
    }

    /**
     * 创建自定义的 Key
     * @param keys Key 数组
     * @return 自定义的 KeyGenerator
     */
    public KeyGenerator generator(String...keys) {
        return this.generator(true, keys);
    }

    /**
     * 创建自定义的 Key
     * @param onNamespace 是否包含命名空间
     * @param keys Key 数组
     * @return 自定义的 KeyGenerator
     */
    public KeyGenerator generator(boolean onNamespace, String...keys) {
        List<Object> list = Lists.newArrayList("game");
        if (onNamespace) {
            list.add(appConfig.getNamespace());
        }
        list.addAll(Arrays.asList(keys));
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户永久资产
     * @return 用户资产 KeyGenerator
     */
    public KeyGenerator getUserAsset() {
        return this.getUserAsset(true);
    }

    /**
     * 用户永久资产
     * @return 用户资产 KeyGenerator
     */
    public KeyGenerator getUserAsset(boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfig.getNamespace(), "assets");
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户永久数据
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getPermanentDataForUser() {
        return this.getPermanentDataForUser(true);
    }

    /**
     * 用户永久数据
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getPermanentDataForUser(boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfig.getNamespace(), "user", "0");
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户当日数据
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getTodayDataForUser() {
        return this.getTodayDataForUser(true);
    }

    /**
     * 用户当日数据
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getTodayDataForUser(boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfig.getNamespace(), "user", DateHelper.getDate());
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 临时数据
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTemporary() {
        return this.getTemporary(true);
    }

    /**
     * 临时数据
     * @param withApp 是否包含应用
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTemporary(boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfig.getNamespace(), "temporary");
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 时效数据
     * @param anchor 锚定日期
     * @param type 时效类目
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTimeBound(String anchor, String type) {
        return this.getTimeBound(anchor, type, true);
    }

    /**
     * 时效数据
     * @param anchor 锚定日期
     * @param type 时效类目
     * @param withApp 是否包含应用
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTimeBound(String anchor, String type, boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfig.getNamespace(), "bound");
        if(null != anchor){
            list.add(anchor);
        }
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        list.add(type);
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户活跃
     * @param withApp 是否包含应用
     * @return 用户活跃
     * @since 1.2.4
     */
    public KeyGenerator bitmapActive(boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfig.getNamespace(), "active");
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户连续签到次数
     * @return 签到次数 KeyGenerator
     */
    public KeyGenerator getSignin() {
        return () -> KeyBuilder.build("game", appConfig.getNamespace(), "signin", "times", AppContext.get().getAppId(), AppContext.get().getUid());
    }

    /**
     * 应用当日数据
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getTodayDataForApp() {
        return () -> KeyBuilder.build("game", appConfig.getNamespace(), "app", DateHelper.getDate(), AppContext.get().getAppId());
    }

    /**
     * 获取调度作业锁
     *
     * @param jobName 作业名称
     * @return 调度作业锁 KeyGenerator
     */
    public KeyGenerator getJobLock(String jobName) {
        return () -> KeyBuilder.build("game", appConfig.getNamespace(), "lock", jobName);
    }

    /**
     * ### 所有游戏共享该数据 ###
     * 最近玩的游戏列表(设备id为准)<br/>
     *
     * @return 最近玩的游戏 KeyGenerator
     */
    public KeyGenerator getLatelyGames() {
        List<Object> list = Lists.newArrayList("game", "common", "lately");
        list.add(AppContext.get().getAppId());
        list.add(AppContext.get().getDevUid());
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * ### 所有游戏共享该数据 ###
     * 游戏列表<br/>
     *
     * @return 游戏信息 KeyGenerator
     */
    public KeyGenerator getGameAll() {
        return () -> KeyBuilder.build("game", "common", "all");
    }
}
