package com.bxm.game.common.core.controllers;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.archives.CustomArchiveHandler;
import com.bxm.game.common.core.archives.CustomArchiveHandlerFactory;
import com.bxm.game.common.core.assets.AssetService;
import com.bxm.game.common.core.scene.signin.SigninService;
import com.bxm.game.common.core.user.*;
import com.bxm.game.common.core.vo.*;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author allen
 * @date 2020-11-30
 * @since 1.0
 */
@Slf4j
@RestController
@RequestMapping("/archive")
public class ArchiveController {

    private final AppConfigFetcher appConfigFetcher;
    private final CustomArchiveFetcherFactory customArchiveFetcherFactory;
    private final CustomArchiveHandlerFactory customArchiveHandlerFactory;
    private final DataTransfer dataTransfer;
    private final AssetArchiveTransfer assetArchiveTransfer;
    private final TodayAtomicService todayAtomicService;
    private final PermanentAtomicService permanentAtomicService;
    private final DefaultTimeBoundService defaultTimeBoundService;
    private final AssetService assetService;
    private final UserActiveService userActiveService;

    @Autowired(required = false)
    private SigninService signinService;
    @Autowired(required = false)
    private LatelyGameService latelyGameService;

    public ArchiveController(AppConfigFetcher appConfigFetcher, CustomArchiveFetcherFactory customArchiveFetcherFactory,
                             CustomArchiveHandlerFactory customArchiveHandlerFactory, DataTransfer dataTransfer, AssetArchiveTransfer assetArchiveTransfer, TodayAtomicService todayAtomicService,
                             PermanentAtomicService permanentAtomicService, DefaultTimeBoundService defaultTimeBoundService,
                             AssetService assetService, UserActiveService userActiveService) {
        this.appConfigFetcher = appConfigFetcher;
        this.customArchiveFetcherFactory = customArchiveFetcherFactory;
        this.customArchiveHandlerFactory = customArchiveHandlerFactory;
        this.dataTransfer = dataTransfer;
        this.assetArchiveTransfer = assetArchiveTransfer;
        this.todayAtomicService = todayAtomicService;
        this.permanentAtomicService = permanentAtomicService;
        this.defaultTimeBoundService = defaultTimeBoundService;
        this.assetService = assetService;
        this.userActiveService = userActiveService;
    }

    @GetMapping("/fetch")
    public ResponseEntity<ArchiveVo> fetch() {
        Map<String, Object> custom = Maps.newHashMap();
        ImmutableSet<CustomArchiveFetcher> fetchers = customArchiveFetcherFactory.getFetchers();
        for (CustomArchiveFetcher fetcher : fetchers) {
            String key = fetcher.getKey();
            if (custom.containsKey(key)) {
                log.warn("duplicated key: {}", key);
            }
            custom.put(key, fetcher.getValue());
        }

        ImmutableSet<CustomArchiveHandler> handlers = customArchiveHandlerFactory.getHandlers();
        for (CustomArchiveHandler handler : handlers) {
            String displayFieldName = handler.getDisplayFieldName();
            if (custom.containsKey(displayFieldName)) {
                log.warn("duplicated key: {}", displayFieldName);
            }
            custom.put(displayFieldName, handler.getArchiveObjects());
        }

        ArchiveVo.ArchiveVoBuilder builder = ArchiveVo.builder()
                .today(dataTransfer.transfer(DataTypeEnum.TODAY,todayAtomicService.getAll()))
                .permanent(dataTransfer.transfer(DataTypeEnum.PERMANENT,permanentAtomicService.getAll()))
                .bound(dataTransfer.transfer(DataTypeEnum.BOUND,defaultTimeBoundService.getAll()))
                .assets(assetArchiveTransfer.transfer(assetService.getAll()))
                .custom(custom)
                .time(System.currentTimeMillis());

        if (Objects.nonNull(signinService)) {
            builder.continuousSignInDays(signinService.getContinuousSignInTimes());
        }
        if (appConfigFetcher.activatingOnFetchArchive()) {
            userActiveService.active();
        }
        if (appConfigFetcher.showUserActiveDays()) {
            builder.activeDays(userActiveService.getActiveDays());
        }

        if(appConfigFetcher.markLatelyGame()){
            //标记为最近玩的游戏
            latelyGameService.markLatelyGame();
        }

        return ResponseEntity.ok(builder.build());
    }

    /**
     * 最近玩的游戏
     * @return
     */
    @GetMapping("/lately")
    public ResponseEntity<List<GameVo>> getLatelyGames(Integer days, Boolean includeSelf) {
        if(null == days){
            days = 30;
        }
        if(null == includeSelf){
            includeSelf = Boolean.FALSE;
        }
        return ResponseEntity.ok(latelyGameService.getLatelyGames(days, includeSelf));
    }
}
