package com.bxm.game.common.core.interceptor;

import com.bxm.game.common.core.AppContext;
import com.bxm.game.common.core.RequestHeaderHandler;
import com.bxm.game.common.core.user.dao.UserDao;
import com.bxm.warcar.integration.dc.dot.Device;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author allen
 * @date 2020-10-14
 * @since 1.0
 */
@Slf4j
public class AppContextInterceptor implements HandlerInterceptor {

    private final UserDao userDao;
    private final RequestHeaderHandler requestHeaderHandler;

    public AppContextInterceptor(UserDao userDao, RequestHeaderHandler requestHeaderHandler){
        this.userDao = userDao;
        this.requestHeaderHandler = requestHeaderHandler;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        String appid = request.getHeader("x-appid");
        String device = request.getHeader("x-device");
        String did = request.getHeader("x-did");
        if (StringUtils.isBlank(appid) || StringUtils.isBlank(device)) {
            log.info("Required header name: 'appid' and 'device' is blank!");
            response.setStatus(HttpStatus.ACCEPTED.value());
            return false;
        }

        AppContext context = AppContext.getInstance();
        if (StringUtils.isNotBlank(did)) {
            Device deviceObject = Device.decrypt(did);
            context.setDevice(deviceObject);

            // UID 之前使用的是明文 imei / android id，国家规定不能非法收集用户隐私，IMEI 属于。
            // 所以在这之后 Android端 UID使用不可逆密文，为了兼容用户的历史数据，需要根据 did 解密反查出当时的UID值。
            // modified by Allen
            if (requestHeaderHandler.isOverrideDeviceValue(appid, device, deviceObject)) {
                String newDevice = requestHeaderHandler.fetchDeviceValue(deviceObject);
                if (!StringUtils.equals(newDevice, device)) {
                    log.warn("{} Not equals, device is:{}, except: {}, but: {}", appid, JsonHelper.convert(deviceObject), device, newDevice);
                }
                // 从Device里获取原来的 UID
                if (StringUtils.isNotBlank(newDevice)) {
                    device = newDevice;
                }
            }
        }

        String uid = userDao.getUid(appid, device);
        if (StringUtils.isBlank(uid)) {
            log.info("uid is blank,appid={},device={}", appid, device);
            response.setStatus(HttpStatus.ACCEPTED.value());
            return false;
        }

        context.setAppId(appid).setDevUid(device).setUid(uid).setDid(did);

        userDao.handleAppContext(context);
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        AppContext.remove();
    }
}
