package com.bxm.game.common.core.job;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bxm.game.common.core.AppConfig;
import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.Consts;
import com.bxm.game.common.core.Key;
import com.bxm.game.common.dal.entity.AssetsSnapshot;
import com.bxm.game.common.dal.service.IAssetsSnapshotService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.RedisLock;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.*;

/**
 * 定时处理资产<br/>
 *
 * @author kerry.jiang
 * @date 2021/1/13 16:05
 */
@Slf4j
public class AssetJobServiceImpl implements AssetJobService {

    private final Fetcher fetcher;
    private final AppConfig appConfig;
    private final AppConfigFetcher appConfigFetcher;
    private final IAssetsSnapshotService iAssetsSnapshotService;
    private final Key key;
    private final RedisLock redisLock;

    public AssetJobServiceImpl(@Qualifier("jedisFetcher") Fetcher fetcher,
                               AppConfig appConfig,AppConfigFetcher appConfigFetcher,
                               IAssetsSnapshotService iAssetsSnapshotService, Key key,
                               @Qualifier("distributedRedisLock") RedisLock redisLock) {
        this.fetcher = fetcher;
        this.appConfig = appConfig;
        this.appConfigFetcher = appConfigFetcher;
        this.iAssetsSnapshotService = iAssetsSnapshotService;
        this.key = key;
        this.redisLock = redisLock;
    }

    /**
     * 作业名称
     */
    private String synAssetJobName = "synAsset";
    /**
     * 作业锁的最长时间（毫秒）
     */
    private int synAssetJobLockTimes = 1000 * 60 * 5;

    @Override
    public void synAsset(Integer lockTimes) {
        int expireTime = null != lockTimes ? lockTimes : synAssetJobLockTimes;
        //加锁
        String lockKey = key.getJobLock(synAssetJobName).generateKey();
        boolean isLock = redisLock.tryLock(lockKey, synAssetJobName, expireTime);
        if (!isLock) {
            throw new RuntimeException("请勿频繁操作");
        }
        new Thread(()->{
            try {
                synAsset0();
            } catch(Exception e){
                log.error("synAsset error:", e);
            } finally {
                //解锁
                redisLock.unLock(lockKey, synAssetJobName);
            }
        }).start();
    }

    private void synAsset0(){
        boolean isSynAppUid = appConfigFetcher.isSynAppUid();
        List<String> assetTypes = appConfigFetcher.synAssetTypes();
        if(CollectionUtils.isEmpty(assetTypes)){
            return;
        }
        LocalDateTime currentDate = LocalDateTime.now();
        String rptDate = DateHelper.format(DateHelper.PATTERN_STR8);
        final String pattern = new StringBuilder().append("*game:").append(appConfig.getNamespace())
                .append(":assets:*:*").toString();

        //删除当日该活动类型的数据
        QueryWrapper<AssetsSnapshot> query = Wrappers.query(new AssetsSnapshot()
                .setRptDate(rptDate)
                .setActivityType(appConfigFetcher.activityType())
        );
        iAssetsSnapshotService.remove(query);

        try (final Jedis jedis = getJedisPool().getResource()) {
            final String startCursor = "0";
            String cursor = startCursor;
            AtomicLong index = new AtomicLong();
            final int count = 10000;
            final ScanParams params = new ScanParams().match(pattern).count(count);
            List<AssetsSnapshot> snapshots = Lists.newArrayList();
            Map<String, AssetsSnapshot> snapshotMap = isSynAppUid ? Maps.newHashMap() : null;

            do {
                final ScanResult<String> scanResult = jedis.scan(cursor, params);
                cursor = scanResult.getStringCursor();
                final List<String> result = scanResult.getResult();
                if (CollectionUtils.isEmpty(result)) {
                    continue;
                }
                for (String k : result) {
                    //填充快照信息
                    fillSnapshot(currentDate, rptDate, snapshots,
                            snapshotMap, k, assetTypes);

                    if(null != snapshotMap){
                        //填充appUid
                        fillAppUid(snapshotMap);
                    }

                    //分批次插入
                    if(snapshots.size() > 199){
                        iAssetsSnapshotService.saveBatch(snapshots);

                        //重置数据
                        snapshots = new ArrayList<>();
                        if(null != snapshotMap){
                            snapshotMap.clear();
                        }
                        currentDate = LocalDateTime.now();
                    }
                }
                log.info("index={} | cursor={} success insert to db: {}",
                        index.addAndGet(count), cursor, snapshots.size());
            } while (! cursor.equals(startCursor));

            if(snapshots.size() > 0){
                iAssetsSnapshotService.saveBatch(snapshots);
            }
        }
    }

    /**
     * 填充快照信息
     */
    private void fillSnapshot(LocalDateTime currentDate, String rptDate, List<AssetsSnapshot> snapshots,
                              Map<String, AssetsSnapshot> snapshotMap, String k, List<String> assetTypes) {
        final Map<String, String> map = fetcher.hfetchall(() -> k, String.class);
        Matcher m = Consts.PATTERN_ASSET.matcher(k);
        if(!m.find()){
            //没有appid/uid
            return ;
        }
        String appid = m.group(1);
        String uid = m.group(2);
        AssetsSnapshot snapshot = new AssetsSnapshot();
        for (String assetType : assetTypes){
            Integer assetNum = Integer.valueOf(map.getOrDefault(assetType, "0"));
            snapshot.setRptDate(rptDate);
            snapshot.setAppId(appid);
            snapshot.setUid(uid);
            snapshot.setAppUid(StringUtils.EMPTY);
            snapshot.setActivityType(appConfigFetcher.activityType());
            snapshot.setAssetType(assetType);
            snapshot.setAssetNum(assetNum);
            snapshot.setCreateTime(currentDate);
            snapshots.add(snapshot);
            if(null != snapshotMap){
                snapshotMap.put(k, snapshot);
            }
        }
    }

    /**
     * 填充appUid
     */
    private void fillAppUid(Map<String, AssetsSnapshot> snapshotMap) {
        Map<String, Response<String>> mapUserRs = new HashMap<>();
        Set<String> keys = snapshotMap.keySet();

        try (Jedis jedis = getJedisPool().getResource()) {
            //切换数据库
            jedis.select(1);

            Pipeline pipeline = jedis.pipelined();
            for (String key : keys) {
                AssetsSnapshot snapshot = snapshotMap.get(key);

                mapUserRs.put(key, pipeline.hget(KeyBuilder.build("ADX", "USERMAPPING", snapshot.getAppId()),
                        snapshot.getUid()));
            }
            pipeline.syncAndReturnAll();
        }

        for (String key : keys) {
            Response<String> rs = mapUserRs.get(key);
            if(null == rs){
                continue;
            }
            String appUid = rs.get();
            if(null != appUid){
                snapshotMap.get(key).setAppUid(appUid);
            }
        }
    }

    private JedisPool getJedisPool() {
        final Object original = fetcher.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            throw new RuntimeException("originalClient is not JedisPool!");
        }
        return (JedisPool) original;
    }
}
