package com.bxm.game.common.core.prop.dao;

import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.prop.AcquiredPropLog;
import com.bxm.game.common.dal.service.IAcquiredPropLogService;
import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;

import java.util.Objects;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @date 2020-12-15
 * @since 1.0
 */
@Slf4j
@ConditionalOnProperty(name = "game.datasource.type", havingValue = "db")
public class DbPropDaoImpl implements PropDao, ApplicationListener<ApplicationStartedEvent> {

    private final AppConfigFetcher appConfigFetcher;
    private final ScheduledThreadPoolExecutor scheduled = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("createTable"));
    private final IAcquiredPropLogService acquiredPropLogService;

    public DbPropDaoImpl(AppConfigFetcher appConfigFetcher, IAcquiredPropLogService acquiredPropLogService) {
        this.appConfigFetcher = appConfigFetcher;
        this.acquiredPropLogService = acquiredPropLogService;
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        log.info("Starting execute create table scheduler.");
        createIfNecessary(0);

        scheduled.scheduleWithFixedDelay(() -> {
            createIfNecessary(1);
        }, 0, 1, TimeUnit.DAYS);
    }

    private void createIfNecessary(int daysToAdd) {
        final String tableName = com.bxm.game.common.dal.entity.AcquiredPropLog.getTableName(daysToAdd);
        acquiredPropLogService.createTableIfNecessary(appConfigFetcher.databaseName(), tableName);
    }

    @Override
    public boolean saveTakeLog(AcquiredPropLog log) {
        com.bxm.game.common.dal.entity.AcquiredPropLog e = new com.bxm.game.common.dal.entity.AcquiredPropLog();
        e.setId(log.getId())
                .setAppId(log.getAppId())
                .setUid(log.getUid())
                .setAssetType(log.getAssetType())
                .setSceneType(log.getSceneType())
                .setPropNum(log.getPropNum())
                .setMultipleNum(log.getMultipleNum())
                .setCompletedMultiple(log.isCompletedMultiple())
                .setAcquired(log.isAcquired())
                .setStartAcquireTime(log.getStartAcquireTime())
                .setEndAcquireTime(log.getEndAcquireTime());
        return acquiredPropLogService.saveOrUpdate(e);
    }

    @Override
    public AcquiredPropLog getTakeLog(String id) {
        final com.bxm.game.common.dal.entity.AcquiredPropLog log = acquiredPropLogService.getById(id);
        if (Objects.isNull(log)) {
            return null;
        }
        return AcquiredPropLog.of(log);
    }
}
