package com.bxm.game.common.core.scene.acquire;

import com.bxm.game.common.core.AppContext;
import com.bxm.game.common.core.ErrCode;
import com.bxm.game.common.core.GameException;
import com.bxm.game.common.core.prop.AcquiredPropLog;
import com.bxm.game.common.core.prop.Prop;
import com.bxm.game.common.core.prop.PropBuilder;
import com.bxm.game.common.core.prop.PropService;
import com.bxm.game.common.core.scene.AbstractSceneService;
import com.bxm.game.common.core.scene.AcquiringOrMultiplyPropEvent;
import com.bxm.game.common.core.scene.SceneType;
import com.bxm.game.common.core.user.TemporaryAtomicService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;

import java.util.Map;
import java.util.Objects;

/**
 * 领取之前获取的道具。
 * 当之前获取到的道具 {@link Prop#acquired()} 是非立即领取的。
 * 那么需要将之前获取的道具单号通过这个服务来领取。
 * @author allen
 * @date 2020-11-04
 * @since 1.0
 */
@Slf4j
public class AcquireSceneServiceImpl extends AbstractSceneService<AcquireSceneRequest, AcquireSceneResponse> implements ApplicationListener<ApplicationReadyEvent> {

    private final Map<String, AcquireConditional> condtionalMap = Maps.newHashMap();
    private final PropService propService;
    private final TemporaryAtomicService temporaryAtomicService;
    private final PropBuilder propBuilder;

    public AcquireSceneServiceImpl(PropService propService, TemporaryAtomicService temporaryAtomicService, PropBuilder propBuilder) {
        this.propService = propService;
        this.temporaryAtomicService = temporaryAtomicService;
        this.propBuilder = propBuilder;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        Map<String, AcquireConditional> beans = event.getApplicationContext().getBeansOfType(AcquireConditional.class);
        for (Map.Entry<String, AcquireConditional> e : beans.entrySet()) {
            AcquireConditional bean = e.getValue();
            String conditionSceneType = bean.getSceneType();
            condtionalMap.put(conditionSceneType, bean);
            log.info("Registered AcquireConditional bean: {}", bean);
        }
    }

    @Override
    protected String createId(AcquireSceneRequest request) {
        return request.getId();
    }

    @Override
    protected Prop takeProp(AcquireSceneRequest request, Map<Object, Object> attach) {
        if (Objects.isNull(request)) {
            return null;
        }
        String appId = AppContext.get().getAppId();
        String uid = AppContext.get().getUid();
        String id = request.getId();
        String barrier = getSceneType() + id;
        if (temporaryAtomicService.incrementAndAbove(barrier, 1)) {
            throw new GameException(ErrCode.NOT_MEET_ACQUIRING_CONDITION);
        }
        AcquiredPropLog takeLog = propService.getTakeLog(id);
        if (Objects.isNull(takeLog)) {
            return null;
        }
        if (!(StringUtils.equals(appId, takeLog.getAppId())
                && StringUtils.equals(uid, takeLog.getUid())
                && !takeLog.isAcquired()
                && takeLog.isBetweenAcquireTime())) {
            return null;
        }
        if (condtionalMap.containsKey(takeLog.getSceneType())
                && !condtionalMap.get(takeLog.getSceneType()).checked(takeLog)) {
            // rollback barrier times
            temporaryAtomicService.incrementBy(barrier, -1);
            throw new GameException(ErrCode.NOT_MEET_ACQUIRING_CONDITION);
        }
        Prop prop = propBuilder.rebuild(takeLog.getAssetType(), takeLog.getPropNum(), takeLog.getMultipleNum());
        if (Objects.nonNull(prop)) {
            propService.saveTakeLog(takeLog.setAcquired(true));
        }
        return prop;
    }

    @Override
    protected void afterCompletion(AcquireSceneRequest request, AcquireSceneResponse response, Prop prop, Map<Object, Object> attach) {
        getSyncEventPark().post(new AcquiredSceneEvent(this, request, response));
        getSyncEventPark().post(new AcquiringOrMultiplyPropEvent(this, request, response));
    }

    @Override
    protected AcquireSceneResponse createResponse(AcquireSceneRequest request, String id, Prop prop, Map<Object, Object> attach) {
        return new AcquireSceneResponse();
    }

    @Override
    public String getSceneType() {
        return SceneType.ACQUIRE;
    }

    @Override
    public Class<AcquireSceneRequest> getRequestClass() {
        return AcquireSceneRequest.class;
    }
}
