package com.bxm.game.common.core.scene.multiple;

import com.bxm.game.common.core.AppContext;
import com.bxm.game.common.core.ErrCode;
import com.bxm.game.common.core.GameException;
import com.bxm.game.common.core.prop.AcquiredPropLog;
import com.bxm.game.common.core.prop.Prop;
import com.bxm.game.common.core.prop.PropBuilder;
import com.bxm.game.common.core.prop.PropService;
import com.bxm.game.common.core.scene.AbstractSceneService;
import com.bxm.game.common.core.scene.AcquiringOrMultiplyPropEvent;
import com.bxm.game.common.core.scene.SceneType;
import com.bxm.game.common.core.user.TemporaryAtomicService;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;

import java.util.Map;
import java.util.Objects;

/**
 * 翻倍场景
 *
 * @author allen
 * @date 2020-11-26
 * @since 1.0
 */
@Slf4j
public class MultipleSceneServiceImpl extends AbstractSceneService<MultipleSceneRequest, MultipleSceneResponse> implements ApplicationListener<ApplicationReadyEvent> {

    private final Map<String, MultipleConditional> condtionalMap = Maps.newHashMap();
    private final PropService propService;
    private final TemporaryAtomicService temporaryAtomicService;
    private final PropBuilder propBuilder;

    public MultipleSceneServiceImpl(PropService propService, TemporaryAtomicService temporaryAtomicService, PropBuilder propBuilder) {
        this.propService = propService;
        this.temporaryAtomicService = temporaryAtomicService;
        this.propBuilder = propBuilder;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        Map<String, MultipleConditional> beans = event.getApplicationContext().getBeansOfType(MultipleConditional.class);
        for (Map.Entry<String, MultipleConditional> e : beans.entrySet()) {
            MultipleConditional bean = e.getValue();
            String conditionSceneType = bean.getSceneType();
            condtionalMap.put(conditionSceneType, bean);
            log.info("Registered MultipleConditional bean: {}", bean);
        }
    }

    @Override
    public boolean isMultipleGrant() {
        return true;
    }

    @Override
    protected String createId(MultipleSceneRequest request) {
        return request.getId();
    }

    @Override
    protected Prop takeProp(MultipleSceneRequest request, Map<Object, Object> attach) {
        if (Objects.isNull(request)) {
            return null;
        }
        String appId = AppContext.get().getAppId();
        String uid = AppContext.get().getUid();
        String id = request.getId();
        String barrier = getSceneType() + id;
        if (temporaryAtomicService.incrementAndAbove(barrier, 1)) {
            throw new GameException(ErrCode.NOT_MEET_ACQUIRING_CONDITION);
        }
        AcquiredPropLog takeLog = propService.getTakeLog(id);
        if (Objects.isNull(takeLog)) {
            return null;
        }
        if (!(StringUtils.equals(appId, takeLog.getAppId())
                && StringUtils.equals(uid, takeLog.getUid())
//                && !takeLog.isAcquired()
                && !takeLog.isCompletedMultiple()
                && takeLog.getMultipleNum() > 1)) {
            return null;
        }
        if (condtionalMap.containsKey(takeLog.getSceneType())
                && !condtionalMap.get(takeLog.getSceneType()).checked(takeLog)) {
            // rollback barrier times
            temporaryAtomicService.incrementBy(barrier, -1);
            throw new GameException(ErrCode.NOT_MEET_MULTIPLE_CONDITION);
        }

        boolean acquired = takeLog.isAcquired();
        int multipleNum = acquired ? takeLog.getMultipleNum() - 1 : takeLog.getMultipleNum();
        Prop prop = propBuilder.rebuild(takeLog.getAssetType(), takeLog.getPropNum(), multipleNum);
        if (Objects.isNull(prop)) {
            return null;
        }
        propService.saveTakeLog(takeLog.setAcquired(true).setCompletedMultiple(true));
        int acquiredNum = prop.propNum() * prop.multipleNum();
        if (acquired) {
            acquiredNum = prop.propNum() * (prop.multipleNum() + 1);
        }
        attach.put("acquiredNum", acquiredNum);
        return prop;
    }

    @Override
    protected MultipleSceneResponse createResponse(MultipleSceneRequest request, String id, Prop prop, Map<Object, Object> attach) {
        MultipleSceneResponse response = new MultipleSceneResponse();
        response.setAcquiredNum(TypeHelper.castToInt(attach.get("acquiredNum")));
        return response;
    }

    @Override
    protected void afterCompletion(MultipleSceneRequest request, MultipleSceneResponse response, Prop prop, Map<Object, Object> attach) {
        getSyncEventPark().post(new MultipleSceneEvent(this, request, response));
        getSyncEventPark().post(new AcquiringOrMultiplyPropEvent(this, request, response));
    }

    @Override
    public String getSceneType() {
        return SceneType.MULTIPLE;
    }

    @Override
    public Class<MultipleSceneRequest> getRequestClass() {
        return MultipleSceneRequest.class;
    }
}
