package com.bxm.game.common.core.scene.signin;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;

import com.alibaba.fastjson.util.TypeUtils;
import com.bxm.game.common.core.Key;
import com.bxm.game.common.core.prop.Prop;
import com.bxm.game.common.core.scene.AbstractMaximumTimesOnDailySceneService;
import com.bxm.game.common.core.scene.SceneType;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.TypeHelper;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * 签到
 *
 * @author allen
 * @date 2020-11-27
 * @since 1.0
 */
@Slf4j
@ConditionalOnBean(SigninConfig.class)
public class SigninSceneServiceImpl extends AbstractMaximumTimesOnDailySceneService<SigninSceneRequest, SigninSceneResponse> implements SigninService {

    private final Key key;
    private final SigninConfig signinConfig;
    private final Counter counter;

    public SigninSceneServiceImpl(Key key, SigninConfig signinConfig, @Qualifier("jedisCounter") Counter counter) {
        this.key = key;
        this.signinConfig = signinConfig;
        this.counter = counter;
    }

    @Override
    public int getContinuousSignInTimes() {
        return TypeHelper.castToInt(Optional.ofNullable(counter.get(key.getSignin())).orElse(0L));
    }

    @Override
    protected Prop takeProp(SigninSceneRequest request, Map<Object, Object> attach) {
        KeyGenerator keyGenerator = key.getSignin();
        long times = Optional.ofNullable(counter.incrementAndGet(keyGenerator)).orElse(1L);
        if (times >= 7) {
            // 今天是第七天
            counter.decrementByAndGet(keyGenerator, times);
        }
        // 设置后天0点失效
        this.expireAtTomorrow(keyGenerator.generateKey());
        Map<Integer, Prop> signinProps = signinConfig.getSigninProps();
        attach.put("days", times);
        return signinProps.get(TypeUtils.castToInt(times));
    }

    @Override
    protected SigninSceneResponse createResponse(SigninSceneRequest request, String id, Prop prop, Map<Object, Object> attach) {
        SigninSceneResponse response = new SigninSceneResponse();
        response.setDays(NumberUtils.toInt(Objects.toString(attach.get("days"))));
        return response;
    }

    @Override
    public String getSceneType() {
        return SceneType.SIGNIN;
    }

    @Override
    public Class<SigninSceneRequest> getRequestClass() {
        return SigninSceneRequest.class;
    }

    private void expireAtTomorrow(String key) {
        Object clientOriginal = counter.getClientOriginal();
        if (clientOriginal instanceof JedisPool) {
            JedisPool jedisPool = (JedisPool) clientOriginal;
            try (Jedis jedis = jedisPool.getResource()) {
                Long at = jedis.expireAt(key, getAfterTomorrowInMillis() / 1000);
                if (Optional.ofNullable(at).orElse(0L) == 0) {
                    log.warn("{} expireAt command execute fail!", key);
                }
            }
        }
    }

    private long getAfterTomorrowInMillis() {
        LocalDate now = LocalDate.now();
        LocalDate tomorrow = now.plusDays(2);
        LocalDateTime time = LocalDateTime.of(tomorrow, LocalTime.of(0, 0));
        return time.toInstant(ZoneOffset.of("+8")).toEpochMilli();
    }
}
