package com.bxm.game.common.core.system;

import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.ErrCode;
import com.bxm.game.common.core.GameException;
import com.bxm.warcar.integration.pair.MutablePair;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

/**
 * @author allen
 * @date 2021-01-29
 * @since 1.0
 */
@Slf4j
public class LocalDateRuntimeClockImpl implements RuntimeClock, InitializingBean {

    private final MutablePair mutablePair;
    private final AppConfigFetcher appConfigFetcher;

    public LocalDateRuntimeClockImpl(MutablePair mutablePair, AppConfigFetcher appConfigFetcher) {
        this.mutablePair = mutablePair;
        this.appConfigFetcher = appConfigFetcher;
    }

    @Override
    public void afterPropertiesSet() {
        String value = mutablePair.get(getKey()).of();
        if (StringUtils.isBlank(value)) {
            initStartingDate();
        }
        long runtimeDays = getRuntimeDays();
        if (runtimeDays < 0) {
            throw new RuntimeException("系统上线天数不能小于 0，当前：" + runtimeDays);
        }
        log.info("{} 系统已上线 {} 天", appConfigFetcher.namespace(), runtimeDays);
    }

    @Override
    public long getRuntimeDays() {
        String value = mutablePair.get(getKey()).of();
        if (StringUtils.isBlank(value)) {
            initStartingDate();
        }
        try {
            LocalDate now = LocalDate.now();
            LocalDate firstDay = LocalDate.parse(value, DateTimeFormatter.ofPattern("yyyyMMdd"));
            return now.toEpochDay() - firstDay.toEpochDay();
        } catch (DateTimeParseException e) {
            log.error("parse: ", e);
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        } catch (Exception e) {
            log.error("err: ", e);
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }
    }

    private void initStartingDate() {
        String date = appConfigFetcher.runtime().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
        mutablePair.set(getKey(), date);
        log.info("设置系统初始化上线日期：{}", date);
    }

    private String getKey() {
        return appConfigFetcher.namespace() + ".app.online.date";
    }
}
