package com.bxm.game.common.core.user.redis;

import java.util.*;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;

import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.AppContext;
import com.bxm.game.common.core.Key;
import com.bxm.game.common.core.user.DefaultTimeBoundService;
import com.bxm.game.common.core.vo.RedisStorageEnum;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.JedisPool;

@Slf4j
@ConditionalOnMissingBean(DefaultTimeBoundService.class)
public class RedisDefaultTimeBoundServiceImpl implements DefaultTimeBoundService {

    /**
     * 时效类目 -> 存储类型
     */
    private static final Map<String, RedisStorageEnum> TYPE_MAP = new HashMap<>();
    /**
     * 不将锚点日期作为key的时效类目
     */
    private List<String> IGNORE_BOUND_ANCHOR = new ArrayList<>();

    protected final AppConfigFetcher appConfigFetcher;
    protected final Key key;
    protected final Fetcher fetcher;
    protected final Counter counter;
    protected final Updater updater;

    public RedisDefaultTimeBoundServiceImpl(AppConfigFetcher appConfigFetcher, Key key,
                                            @Qualifier("jedisFetcher") Fetcher fetcher,
                                            @Qualifier("jedisUpdater") Updater updater,
                                            @Qualifier("jedisCounter") Counter counter) {
        this.appConfigFetcher = appConfigFetcher;
        this.key = key;
        this.fetcher = fetcher;
        this.counter = counter;
        this.updater = updater;

        Map<String, RedisStorageEnum> m = getTypeMap();
        if(null != m){
            TYPE_MAP.putAll(m);
        }
        List<String> list = ignoreBoundAnchorTypes();
        if(null != list){
            IGNORE_BOUND_ANCHOR = list;
        }
    }

    private JedisPool getJedisPool() {
        final Object original = fetcher.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            throw new RuntimeException("originalClient is not JedisPool!");
        }
        return (JedisPool) original;
    }

    /**
     * 获取redis key
     */
    private KeyGenerator getKey(String type){
        if(IGNORE_BOUND_ANCHOR.contains(type) || null == AppContext.get().getBoundAnchor()) {
            return this.key.getTimeBound(null, type, appConfigFetcher.timeBoundWithApp());
        }else{
            return this.key.getTimeBound(AppContext.get().getBoundAnchor(), type,
                    appConfigFetcher.timeBoundWithApp());
        }
    }

    @Override
    public Map<String, Object> getAll() {
        Map<String, Object> map = new HashMap();
        Set<String> types = TYPE_MAP.keySet();
        for (String type : types) {
            RedisStorageEnum storageEnum = TYPE_MAP.get(type);

            Object obj = null;
            if(RedisStorageEnum.HASH == storageEnum){
                obj = fetcher.hfetchall(getKey(type), Object.class);
            }else{
                obj = fetcher.fetch(getKey(type), Object.class);
            }
            map.put(type, obj);
        }
        return map;
    }

    @Override
    public Map<String, RedisStorageEnum> getTypeMap() {
        Map<String, RedisStorageEnum> map = new HashMap<>();
        map.put(TYPE_ANCHOR, RedisStorageEnum.OBJECT);
        map.put(TYPE_FREQ, RedisStorageEnum.HASH);
        map.put(TYPE_ASSET, RedisStorageEnum.HASH);
        return map;
    }

    @Override
    public long get(String type, long defaultValue) {
        return Optional.ofNullable(counter.get(getKey(type))).orElse(defaultValue);
    }

    @Override
    public <T> T get(String type, Class<T> cls) {
        return fetcher.fetch(getKey(type), cls);
    }

    @Override
    public void set(String type, String value, int ttl) {
        updater.update(getKey(type), value, ttl);
    }

    @Override
    public long incrBy(String type, long value, int ttl) {
        return Optional.ofNullable(counter.incrementByAndGet(getKey(type), value, ttl)).orElse(0L);
    }

    @Override
    public boolean incrBy(String type, long value, Predicate<Long> predicate, int ttl) {
        long l = this.incrBy(type, value);
        return predicate.test(l);
    }

    @Override
    public void del(String type) {
        updater.remove(getKey(type));
    }

    @Override
    public long hGet(String type, String field, long defaultValue) {
        return Optional.ofNullable(counter.hget(getKey(type), field)).orElse(defaultValue);
    }

    @Override
    public <T> T hGet(String type, String field, Class<T> cls) {
        return fetcher.hfetch(getKey(type), field, cls);
    }

    @Override
    public Map<String, Object> hGetAll(String type) {
        return fetcher.hfetchall(getKey(type), Object.class);
    }

    @Override
    public <T> Map<String, T> hGetAll(String type, Class<T> cls) {
        return fetcher.hfetchall(getKey(type), cls);
    }

    @Override
    public void hSet(String type, String field, String value, int ttl) {
        updater.hupdate(getKey(type), field, value, ttl);
    }

    @Override
    public long hIncrBy(String type, String field, long value, int ttl) {
        return Optional.ofNullable(counter.hincrementByAndGet(getKey(type), field, value, ttl)).orElse(0L);
    }

    @Override
    public boolean hIncrBy(String type, String field, long value, Predicate<Long> predicate, int ttl) {
        long l = this.hIncrBy(type, field, value, ttl);
        return predicate.test(l);
    }

    @Override
    public void hDel(String type, String field) {
        updater.hremove(getKey(type), field);
    }
}
