package com.bxm.game.common.core;

import java.time.LocalDate;
import java.util.Collections;
import java.util.List;

import com.bxm.game.common.core.job.SynAssetModel;
import com.bxm.game.common.core.scene.listeners.ActiveListener;
import com.google.common.collect.Lists;

/**
 * @author allen
 * @date 2020-12-09
 * @since 1.0
 */
public interface AppConfigFetcher {

    /**
     * 服务是否开放，如果返回 {@code false}，那么所以的请求都会被拒绝。
     * @return 是否开放中
     */
    default boolean isOpening() {
        return true;
    }

    /**
     * 空间名称，如果需要数据在不同的项目里互通，那么请设置成一致的名称。
     * @return 空间名称
     */
    String namespace();

    /**
     * 资产数据是否使用 APP 维度
     *
     * @return 是则APP间隔离，否则不隔离
     * @see com.bxm.game.common.core.assets.AssetService
     */
    default boolean assetWithApp() {
        return true;
    }

    /**
     * 用户永久数据是否使用 APP 维度
     *
     * @return 是则APP间隔离，否则不隔离
     * @see com.bxm.game.common.core.user.PermanentAtomicService
     */
    default boolean permanentWithApp() {
        return true;
    }

    /**
     * 用户今日数据是否使用 APP 维度
     *
     * @return 是则APP间隔离，否则不隔离
     * @see com.bxm.game.common.core.user.TodayAtomicService
     */
    default boolean todayWithApp() {
        return true;
    }

    /**
     * 临时数据是否使用 APP 维度
     *
     * @return  是则APP间隔离，否则不隔离
     * @see com.bxm.game.common.core.user.TemporaryAtomicService
     */
    default boolean temporaryWithApp() {
        return true;
    }

    /**
     * 用户活跃数据是否使用 APP 维度
     * @return 是则APP间隔离，否则不隔离，默认：true
     */
    default boolean activeWithApp() {
        return true;
    }

    /**
     * AppContextInterceptor 排除的路径，不排除的话默认会校验 x-appid 等必要参数。
     * @return 排除的路径
     */
    default List<String> excludePathPatternsOfAppContextInterceptor() {
        return Lists.newArrayList();
    }

    /**
     * 所有拦截器都会排除的路径。
     * @return 排除的路径
     * @since 1.2.5
     */
    default List<String> excludePathPatternsOfGlobal() {
        return Lists.newArrayList("/job/**", "/withdrawlocal/**");
    }

    /**
     * 返回数据库名称
     * @return 数据库名称，默认：{@code activity}
     */
    default String databaseName() {
        return "activity";
    }

    /**
     * 返回活动类型
     *
     * fruits-果园
     * farms-农场
     * mcat-变现猫
     * richest-大富翁
     * ...
     * @return 活动类型
     */
    default String activityType(){
        throw new RuntimeException("请配置活动类型");
    }

    /**
     * 资产同步时，是否同步app用户ID
     *
     * @return 默认：false
     */
    default boolean isSynAppUid(){
        return false;
    }

    /**
     * 资产同步时，同步的资产类型
     *
     * @return 默认空集合
     */
    default List<String> synAssetTypes(){
        return Collections.emptyList();
    }

    /**
     * 资产同步时，同步的数据模块
     * @since 1.2.13
     *
     * @return {@link SynAssetModel#ASSET}
     */
    default List<SynAssetModel> synAssetModels(){
        return Lists.newArrayList(SynAssetModel.ASSET);
    }

    /**
     * 返回系统上线的日期，用于计算用户活跃天数。
     * @return 默认为现在
     * @since 1.2.4
     */
    default LocalDate runtime() {
        return LocalDate.now();
    }

    /**
     * 在存档接口的返回内容中是否展示用户的活跃天数
     * @return 默认：false
     * @since 1.2.4
     */
    default boolean showUserActiveDays() {
        return false;
    }

    /**
     * <p>在加载存档接口时，是否设置今天为活跃的。</p>
     * <p>如果返回 true，那么用户加载存档今天即设置活跃了</p>
     * <p>如果返回 false，那么用户只有在发放道具时才会被设置为活跃</p>
     * @return 默认：true
     * @since 1.2.4
     * @see ActiveListener
     */
    default boolean activatingOnFetchArchive() {
        return true;
    }

    /**
     * 时效数据是否使用 APP 维度
     *
     * @return  默认：true, 是则APP间隔离，否则不隔离
     * @since 1.2.5
     */
    default boolean timeBoundWithApp() {
        return true;
    }

    /**
     * 是否标识该游戏为用户最近玩的游戏
     *
     * @return  默认：false
     * @since 1.2.6
     */
    default boolean markLatelyGame() {
        return false;
    }

    /**
     * 产生资产的日志保存时长（秒）<br/>
     *
     * @return 默认：2天
     * @since 1.2.7
     */
    default int ttlTakeLog(){
        return Key.EXPIRE_TIME_ONE_DAY * 2;
    }
}
