package com.bxm.game.common.core.job;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bxm.game.common.core.AppConfig;
import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.Consts;
import com.bxm.game.common.core.Key;
import com.bxm.game.common.dal.entity.AssetsSnapshot;
import com.bxm.game.common.dal.service.IAssetsSnapshotService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.RedisLock;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.*;

/**
 * 定时处理资产<br/>
 *
 * @author kerry.jiang
 * @date 2021/1/13 16:05
 */
@Slf4j
public class AssetJobServiceImpl implements AssetJobService {

    /**
     * 作业名称
     */
    private final static String JOB_SYN_ASSET = "synAsset";
    /**
     * 作业名称
     */
    private final static String JOB_SYN_BOUND_ASSET = "synBoundAsset";
    /**
     * 作业锁的最长时间（毫秒）
     */
    private final static int JOB_LOCK_TIMES = 1000 * 60 * 5;

    private final Fetcher fetcher;
    private final Fetcher appUidfetcher;
    private final AppConfig appConfig;
    private final AppConfigFetcher appConfigFetcher;
    private final IAssetsSnapshotService iAssetsSnapshotService;
    private final Key key;
    private final RedisLock redisLock;

    public AssetJobServiceImpl(@Qualifier("jedisFetcher") Fetcher fetcher,
                               @Autowired(required = false) @Qualifier("commonFetcher") Fetcher appUidfetcher,
                               AppConfig appConfig,AppConfigFetcher appConfigFetcher,
                               IAssetsSnapshotService iAssetsSnapshotService, Key key,
                               @Qualifier("distributedRedisLock") RedisLock redisLock) {
        this.fetcher = fetcher;
        this.appUidfetcher = appUidfetcher;
        this.appConfig = appConfig;
        this.appConfigFetcher = appConfigFetcher;
        this.iAssetsSnapshotService = iAssetsSnapshotService;
        this.key = key;
        this.redisLock = redisLock;
    }

    @Override
    public void synAsset(SynAssetRequest request) {
        int expireTime = Optional.ofNullable(request.getLockTimes()).orElse(JOB_LOCK_TIMES);
        //加锁
        String lockKey = key.getJobLock(JOB_SYN_ASSET).generateKey();
        boolean isLock = redisLock.tryLock(lockKey, JOB_SYN_ASSET, expireTime);
        if (!isLock) {
            throw new RuntimeException("请勿频繁操作");
        }
        new Thread(()->{
            try {
                if(log.isDebugEnabled()){
                    log.debug("synAsset start");
                }

                synAsset0(request);

                if(log.isDebugEnabled()){
                    log.debug("synAsset end");
                }
            } catch(Exception e){
                log.error("synAsset error:", e);
            } finally {
                //解锁
                redisLock.unLock(lockKey, JOB_SYN_ASSET);
            }
        }).start();
    }

    private void synAsset0(SynAssetRequest request){
        boolean isSynAppUid = appConfigFetcher.isSynAppUid();
        List<String> assetTypes = appConfigFetcher.synAssetTypes();
        if(CollectionUtils.isEmpty(assetTypes)){
            return;
        }
        LocalDateTime currentDate = LocalDateTime.now();
        String rptDate = DateHelper.format(DateHelper.PATTERN_STR8);
        //删除当日该活动类型的数据
        QueryWrapper<AssetsSnapshot> query = Wrappers.query(new AssetsSnapshot()
                .setRptDate(rptDate)
                .setActivityType(appConfigFetcher.activityType())
        );
        iAssetsSnapshotService.remove(query);

        // 同步的数据模块
        List<SynAssetModel> synAssetModels = appConfigFetcher.synAssetModels();
        if(synAssetModels.contains(SynAssetModel.ASSET)){
            // 插入asset模块的资产数据
            insertAssetData(request, isSynAppUid, assetTypes,
                    currentDate, rptDate);
        }
        if(synAssetModels.contains(SynAssetModel.BOUND)){
            // 插入bound模块中的资产信息
            insertBoundAssetData(request, isSynAppUid, assetTypes,
                    currentDate, rptDate);
        }
    }

    /**
     * 插入asset模版的资产数据
     */
    private void insertAssetData(SynAssetRequest request, boolean isSynAppUid, List<String> assetTypes,
                                 LocalDateTime currentDate, String rptDate) {
        // 获取scan所需要的表达式
        // game:farms-api:assets:{appid}:{uid}
        final String pattern = new StringBuilder().append("*game:").append(appConfig.getNamespace())
                .append(":assets:*:*").toString();

        try (final Jedis jedis = getJedisPool().getResource()) {
            final String startCursor = "0";
            String cursor = startCursor;
            AtomicLong index = new AtomicLong();
            final int count = 10000;
            final ScanParams params = new ScanParams().match(pattern).count(count);
            List<AssetsSnapshot> snapshots = Lists.newArrayList();
            Map<String, AssetsSnapshot> snapshotMap = isSynAppUid ? Maps.newHashMap() : null;

            do {
                final ScanResult<String> scanResult = jedis.scan(cursor, params);
                cursor = scanResult.getStringCursor();
                final List<String> result = scanResult.getResult();
                if (CollectionUtils.isEmpty(result)) {
                    continue;
                }
                Map<String, String[]> appMap = new HashMap<>();
                // 填充app/uid信息
                fillAppInfoMap(result, appMap);

                for (String k : result) {
                    //填充快照信息
                    fillSnapshot(currentDate, rptDate, snapshots,
                            snapshotMap, k, assetTypes, appMap);

                    if(Boolean.TRUE.equals(request.getSynAppUid()) && null != snapshotMap){
                        //填充appUid
                        fillAppUid(snapshotMap);
                    }

                    //分批次插入
                    if(snapshots.size() > 199){
                        iAssetsSnapshotService.saveBatch(snapshots);

                        //重置数据
                        snapshots = new ArrayList<>();
                        if(null != snapshotMap){
                            snapshotMap.clear();
                        }
                        currentDate = LocalDateTime.now();
                    }
                }
                log.info("index={} | cursor={} success insert to db: {}",
                        index.addAndGet(count), cursor, snapshots.size());
            } while (! cursor.equals(startCursor));

            if(snapshots.size() > 0){
                iAssetsSnapshotService.saveBatch(snapshots);
            }
        }
    }

    /**
     * 填充app/uid信息
     */
    private void fillAppInfoMap(List<String> result, Map<String, String[]> appMap) {
        for (String k : result) {
            Matcher m = Consts.PATTERN_ASSET.matcher(k);
            if (!m.find()) {
                //没有appid/uid
                continue;
            }
            String appid = m.group(1);
            String uid = m.group(2);
            appMap.put(k, new String[]{appid, uid});
        }
    }

    /**
     * 填充快照信息
     */
    private void fillSnapshot(LocalDateTime currentDate, String rptDate, List<AssetsSnapshot> snapshots,
                              Map<String, AssetsSnapshot> snapshotMap, String k, List<String> assetTypes,
                              Map<String, String[]> appMap) {
        String[] arr = appMap.get(k);
        if(null == arr){
            //没有appid/uid
            return ;
        }
        String appid = arr[0];
        String uid = arr[1];

        Map<String, String> map = fetcher.hfetchall(() -> k, String.class);
        if(null == map){
            map = new HashMap<>();
        }

        AssetsSnapshot snapshot = new AssetsSnapshot();
        for (String assetType : assetTypes){
            Integer assetNum = Integer.valueOf(map.getOrDefault(assetType, "0"));
            snapshot.setRptDate(rptDate);
            snapshot.setAppId(appid);
            snapshot.setUid(uid);
            snapshot.setAppUid(StringUtils.EMPTY);
            snapshot.setActivityType(appConfigFetcher.activityType());
            snapshot.setAssetType(assetType);
            snapshot.setAssetNum(assetNum);
            snapshot.setCreateTime(currentDate);
            snapshots.add(snapshot);
            if(null != snapshotMap){
                snapshotMap.put(k, snapshot);
            }
        }
    }

    /**
     * 填充appUid
     */
    private void fillAppUid(Map<String, AssetsSnapshot> snapshotMap) {
        Map<String, Response<String>> mapUserRs = new HashMap<>();
        Set<String> keys = snapshotMap.keySet();

        try (Jedis jedis = getAppUidJedisPool().getResource()) {
            //切换数据库
            jedis.select(1);

            Pipeline pipeline = jedis.pipelined();
            for (String key : keys) {
                AssetsSnapshot snapshot = snapshotMap.get(key);

                mapUserRs.put(key, pipeline.hget(KeyBuilder.build("ADX", "USERMAPPING", snapshot.getAppId()),
                        snapshot.getUid()));
            }
            pipeline.syncAndReturnAll();
        }

        for (String key : keys) {
            Response<String> rs = mapUserRs.get(key);
            if(null == rs){
                continue;
            }
            String appUid = rs.get();
            if(null != appUid){
                snapshotMap.get(key).setAppUid(appUid);
            }
        }
    }

    private JedisPool getJedisPool() {
        final Object original = fetcher.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            throw new RuntimeException("originalClient is not JedisPool!");
        }
        return (JedisPool) original;
    }

    private JedisPool getAppUidJedisPool() {
        final Object original = appUidfetcher.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            throw new RuntimeException("originalClient is not JedisPool!");
        }
        return (JedisPool) original;
    }

    /**
     * 插入bound模块中的资产信息
     */
    private void insertBoundAssetData(SynAssetRequest request, boolean isSynAppUid, List<String> assetTypes, LocalDateTime currentDate, String rptDate) {
        // 获取scan所需要的表达式
        // game:cow:bound:{appid}:{uid}:anchor
        final String pattern = new StringBuilder().append("*game:").append(appConfig.getNamespace())
                .append(":bound:*:*:anchor").toString();
        try (final Jedis jedis = getJedisPool().getResource()) {
            final String startCursor = "0";
            String cursor = startCursor;
            AtomicLong index = new AtomicLong();
            final int count = 10000;
            final ScanParams params = new ScanParams().match(pattern).count(count);
            List<AssetsSnapshot> snapshots = Lists.newArrayList();
            Map<String, AssetsSnapshot> snapshotMap = isSynAppUid ? Maps.newHashMap() : null;

            do {
                final ScanResult<String> scanResult = jedis.scan(cursor, params);
                cursor = scanResult.getStringCursor();
                final List<String> result = scanResult.getResult();
                if (CollectionUtils.isEmpty(result)) {
                    continue;
                }
                // 资产keys
                List<String> keys = new ArrayList<>();
                Map<String, String[]> appMap = new HashMap<>();
                // 填充资产redis keys
                fillAssetRedisKeys(keys, result, appMap);

                for (String key : keys) {
                    //填充快照信息
                    fillSnapshot(currentDate, rptDate, snapshots,
                            snapshotMap, key, assetTypes, appMap);

                    if(Boolean.TRUE.equals(request.getSynAppUid()) && null != snapshotMap){
                        //填充appUid
                        fillAppUid(snapshotMap);
                    }

                    //分批次插入
                    if(snapshots.size() > 199){
                        iAssetsSnapshotService.saveBatch(snapshots);

                        //重置数据
                        snapshots = new ArrayList<>();
                        if(null != snapshotMap){
                            snapshotMap.clear();
                        }
                        currentDate = LocalDateTime.now();
                    }
                }
                log.info("index={} | cursor={} success insert to db: {}",
                        index.addAndGet(count), cursor, snapshots.size());
            } while (! cursor.equals(startCursor));

            if(snapshots.size() > 0){
                iAssetsSnapshotService.saveBatch(snapshots);
            }
        }
    }

    /**
     * 填充资产redis keys
     */
    private void fillAssetRedisKeys(List<String> keys, List<String> result, Map<String, String[]> appMap) {

        // 获取当前的锚定日期
        Map<String, Response<String>> anchorRsMap = new HashMap<>();
        try (final Jedis jedis = getJedisPool().getResource()) {
            Pipeline pipelined = jedis.pipelined();
            for (String k : result) {
                anchorRsMap.put(k, pipelined.get(k));
            }
            pipelined.syncAndReturnAll();
        }

        for (String k : result) {
            String anchor = anchorRsMap.get(k).get();
            if(StringUtils.isBlank(anchor)){
                // 没有锚定日期
                continue;
            }
            Matcher m = Consts.PATTERN_ANCHOR.matcher(k);
            if (!m.find()) {
                //没有appid/uid
                continue;
            }
            String appid = m.group(1);
            String uid = m.group(2);

            // game:cow:bound:{anchor}:{appid}:{uid}:asset
            String key = KeyBuilder.build(new Object[]{
                    "game",
                    appConfig.getNamespace(),
                    "bound",
                    anchor,
                    appid,
                    uid,
                    "asset"
            });
            appMap.put(key, new String[]{appid, uid});
            keys.add(key);
        }
    }
}
