package com.bxm.game.common.core.user.redis;

import java.util.*;

import org.apache.commons.collections.keyvalue.DefaultKeyValue;
import org.apache.commons.lang.time.DateUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;

import com.bxm.game.common.core.AppConfigFetcher;
import com.bxm.game.common.core.Consts;
import com.bxm.game.common.core.Key;
import com.bxm.game.common.core.user.LatelyGameService;
import com.bxm.game.common.core.vo.GameVo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;

/**
 * 最近玩的游戏<br/>
 *
 * @author kerry.jiang
 * @date 2021/4/1 17:26
 */
@ConditionalOnBean(name = "commonFetcher")
public class RedisLatelyGameServiceImpl implements LatelyGameService {

    protected final AppConfigFetcher appConfigFetcher;
    protected final Key key;
    protected final Fetcher fetcher;
    protected final Updater updater;

    public RedisLatelyGameServiceImpl(AppConfigFetcher appConfigFetcher, Key key,
                                      @Qualifier("commonFetcher") Fetcher fetcher,
                                      @Qualifier("commonUpdater") Updater updater) {
        this.appConfigFetcher = appConfigFetcher;
        this.key = key;
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public void markLatelyGame() {
        double score = (double) System.currentTimeMillis();
        updater.zupdateWithSelector(key.getLatelyGames(), score,
                appConfigFetcher.activityType(),
                Consts.LATELY_GAME_TTL, Consts.LATELY_GAME_DB);
    }

    @Override
    public List<GameVo> getLatelyGames(Integer days, Boolean includeSelf) {
        // 按分数从小到大排序
        long start = 0L;
        if(null != days){
            Calendar cal = Calendar.getInstance();
            cal.set(Calendar.HOUR_OF_DAY, 0);
            cal.set(Calendar.MINUTE, 0);
            cal.set(Calendar.SECOND, 0);
            cal.set(Calendar.MILLISECOND, 0);
            start = DateUtils.addDays(cal.getTime(), -days).getTime();
        }
        List<GameVo> list = new ArrayList<>();
        // 填充游戏信息
        fillGames(list, start, includeSelf);

        if(list.size() > 1){
            // 列表序列反转
            Collections.reverse(list);
        }
        return list;
    }

    /**
     * 填充游戏信息
     */
    private void fillGames(List<GameVo> list, long start, Boolean includeSelf) {
        List<DefaultKeyValue> keyValues = fetcher.zfetchAllWithSelector(key.getLatelyGames(),
                String.class, Consts.LATELY_GAME_DB);
        Map<String, GameVo> gameMap = null;
        for (DefaultKeyValue keyValue : keyValues) {
            String k = (String) keyValue.getKey();
            Double v = (Double) keyValue.getValue();
            if(k.equals(appConfigFetcher.activityType())){
                if(!includeSelf){
                    // 不包含自身
                    continue;
                }
            }
            if(v.longValue() > start){
                if(null == gameMap){
                    gameMap = fetcher.hfetchall(key.getGameAll(), GameVo.class);
                }
                GameVo game = gameMap.get(k);
                if(null != game){
                    list.add(game);
                }
            }
        }
    }
}
