package com.bxm.game.scene.common.core;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.bxm.game.scene.common.core.user.DefaultTimeBoundService;
import com.google.common.collect.Lists;

/**
 * 项目配置信息<br/>
 *
 * @author kerry.jiang
 * @date 2021/10/21 17:59
 */
public interface AppConfigFetcher {

    /**
     * 服务是否开放，如果返回 {@code false}，那么所以的请求都会被拒绝。
     *
     * @return 是否开放中
     */
    default boolean isOpening() {
        return true;
    }

    /**
     * 空间名称，如果需要数据在不同的项目里互通，那么请设置成一致的名称。
     *
     * @return 空间名称
     */
    default String namespace(){
        throw new RuntimeException("请配置命名空间");
    }

    /**
     * 场景类型
     *
     * withdraw-提现
     * ...
     * @return 场景类型
     */
    default String activityType(){
        throw new RuntimeException("请配置活动类型");
    }

    /**
     * 触发debug模式的访问者IP列表<br/>
     *
     * @return
     */
    default List<String> debugIps(){
        return Collections.emptyList();
    }

    /**
     * AppContextInterceptor 排除的路径，不排除的话默认会校验 x-appid 等必要参数。
     *
     * @return 排除的路径
     */
    default List<String> excludePathPatternsOfAppContextInterceptor() {
        return Lists.newArrayList();
    }

    /**
     * 所有拦截器都会排除的路径。
     *
     * @return 排除的路径
     */
    default List<String> excludePathPatternsOfGlobal() {
        return Lists.newArrayList();
    }

    /**
     * 用户永久数据是否使用 APP 维度
     *
     * @param type 时效类目
     * @return 是则APP间隔离，否则不隔离
     * @see com.bxm.game.scene.common.core.user.PermanentAtomicService
     */
    default boolean permanentWithAppForUser(String type) {
        return true;
    }

    /**
     * 用户时效数据是否使用 APP 维度
     *
     * @param type 时效类目
     * @return  默认：true, 是则APP间隔离，否则不隔离
     * @see DefaultTimeBoundService
     */
    default boolean defaultBoundWithAppForUser(String type) {
        return true;
    }

    /**
     * 用户今日数据是否使用 APP 维度
     *
     * @param type 时效类目
     * @return 是则APP间隔离，否则不隔离
     * @see com.bxm.game.scene.common.core.user.TodayAtomicService
     */
    default boolean todayWithAppForUser(String type) {
        return true;
    }

    /**
     * 游戏级今日数据是否使用 APP 维度
     *
     * @param type 时效类目
     * @return 是则APP间隔离，否则不隔离
     * @see com.bxm.game.scene.common.core.user.TodayAtomicService
     */
    default boolean todayWithAppForGame(String type) {
        return true;
    }

    /**
     * 用户临时数据是否使用 APP 维度
     *
     * @param type 时效类目
     * @return  是则APP间隔离，否则不隔离
     * @see com.bxm.game.scene.common.core.user.TemporaryAtomicService
     */
    default boolean temporaryWithAppForUser(String type) {
        return true;
    }

    /**
     * 返回数据库名称
     * @return 数据库名称，默认：{@code activity}
     */
    default String databaseName() {
        return "activity";
    }

    /**
     * 产生资产的日志保存时长（秒）<br/>
     *
     * @return 默认：2天
     */
    default int ttlTakeLog(){
        return DefaultConsts.TTL_DAYS * 2;
    }

    /**
     * 返回系统上线的日期，用于计算用户活跃天数。
     * @return 默认为现在
     */
    default LocalDate runtime() {
        return LocalDate.now();
    }

    /**
     * 在存档接口的返回内容中是否展示用户的活跃天数
     * @return 默认：false
     */
    default boolean showUserActiveDays() {
        return false;
    }

    /**
     * <p>在加载存档接口时，是否设置今天为活跃的。</p>
     * <p>如果返回 true，那么用户加载存档今天即设置活跃了</p>
     * <p>如果返回 false，那么用户只有在发放道具时才会被设置为活跃</p>
     * @return 默认：true
     * @see com.bxm.game.scene.common.core.scene.listeners.ActiveListener
     */
    default boolean activatingOnFetchArchive() {
        return true;
    }

    /**
     * 时效数据需时长延续的类目。<br/>
     * 用户连续N天（默认15天）不活跃，将会清理数据<br/>
     * 中间有活跃时，需要延长时效数据指定类目的ttl值<br/>
     * @return
     */
    default List<String> getBoundRedisKeyTypes(){
        return Arrays.asList(
                DefaultRedisKeyType.SIGN,
                DefaultRedisKeyType.STORAGE,
                DefaultRedisKeyType.ACTIVE,
                DefaultRedisKeyType.FREQ,
                DefaultRedisKeyType.ASSET,
                DefaultRedisKeyType.INFO
        );
    }
}
