package com.bxm.game.scene.common.core;

import java.util.Arrays;
import java.util.List;

import com.bxm.game.scene.common.core.bean.AppContext;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;

/**
 * @author allen
 * @date 2020-10-16
 * @since 1.0
 */
public final class Key {

    public static final int REGISTER_GAME_BUCKETS = 10;
    public static final int REGISTER_APP_BUCKETS = 10;
    public static final int REGISTER_USER_BUCKETS = 1000;
    private final AppConfigFetcher appConfigFetcher;

    public Key(AppConfigFetcher appConfigFetcher) {
        this.appConfigFetcher = appConfigFetcher;
    }

    /**
     * 创建自定义的 Key
     * @param keys Key 数组
     * @return 自定义的 KeyGenerator
     */
    public KeyGenerator generator(String...keys) {
        return this.generator(true, keys);
    }

    /**
     * 创建自定义的 Key
     * @param onNamespace 是否包含命名空间
     * @param keys Key 数组
     * @return 自定义的 KeyGenerator
     */
    public KeyGenerator generator(boolean onNamespace, String...keys) {
        List<Object> list = Lists.newArrayList("game");
        if (onNamespace) {
            list.add(appConfigFetcher.namespace());
        }
        list.addAll(Arrays.asList(keys));
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户永久数据
     * @param type 类目
     * @return 永久数据 KeyGenerator
     */
    public KeyGenerator getPermanentForUser(String type) {
        return this.getPermanentForUser(type, true);
    }

    /**
     * 用户永久数据
     * @param type 类目
     * @return 永久数据 KeyGenerator
     */
    public KeyGenerator getPermanentForUser(String type, boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfigFetcher.namespace(), "user", "0");
        list.add(AppContext.get().getGameCode());
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        list.add(type);
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 用户当日数据
     * @param type 类目
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getTodayForUser(String type) {
        return this.getTodayForUser(type, true);
    }

    /**
     * 用户当日数据
     * @param type 类目
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getTodayForUser(String type, boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfigFetcher.namespace(), "user", DateHelper.getDate());
        list.add(AppContext.get().getGameCode());
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        list.add(type);
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 游戏当日数据
     * @param type 类目
     * @return 当日数据 KeyGenerator
     */
    public KeyGenerator getTodayForGame(String type) {
        return this.getTodayForGame(type, true);
    }

    /**
     * 游戏当日数据
     * @param type 类目
     * @return
     */
    public KeyGenerator getTodayForGame(String type, boolean withApp){
        List<Object> list = Lists.newArrayList("game", appConfigFetcher.namespace(), "game", DateHelper.getDate());
        list.add(AppContext.get().getGameCode());
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(type);
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 临时数据
     * @param type 类目
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTemporaryForUser(String type) {
        return this.getTemporaryForUser(type, true);
    }

    /**
     * 临时数据
     * @param type 类目
     * @param withApp 是否包含应用
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTemporaryForUser(String type, boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfigFetcher.namespace(), "temporary");
        list.add(AppContext.get().getGameCode());
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        list.add(type);
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 时效数据
     * @param anchor 锚定日期
     * @param type 时效类目
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTimeBoundForUser(String anchor, String type) {
        return this.getTimeBoundForUser(anchor, type, true);
    }

    /**
     * 时效数据
     * @param anchor 锚定日期
     * @param type 时效类目
     * @param withApp 是否包含应用
     * @return 临时数据 KeyGenerator
     */
    public KeyGenerator getTimeBoundForUser(String anchor, String type, boolean withApp) {
        List<Object> list = Lists.newArrayList("game", appConfigFetcher.namespace(), "bound");
        if(null != anchor){
            list.add(anchor);
        }
        list.add(AppContext.get().getGameCode());
        if (withApp) {
            list.add(AppContext.get().getAppId());
        }
        list.add(AppContext.get().getUid());
        list.add(type);
        return () -> KeyBuilder.build(list.toArray(new Object[0]));
    }

    /**
     * 注册游戏代码
     * @param bucket 分桶值区间: [0, {@link #REGISTER_GAME_BUCKETS}}
     * @return
     */
    public KeyGenerator getRegisterGames(int bucket){
        return () -> KeyBuilder.build("game", appConfigFetcher.namespace(), "register", "games", bucket);
    }

    /**
     * 注册appid
     * @param bucket 分桶值区间: [0, {@link #REGISTER_APP_BUCKETS}}
     * @return
     */
    public KeyGenerator getRegisterApps(int bucket){
        return () -> KeyBuilder.build("game", appConfigFetcher.namespace(), "register", "apps", bucket);
    }

    /**
     * appid注册的注册用户
     * @param bucket 分桶值区间: [0, {@link #REGISTER_USER_BUCKETS}}
     * @return
     */
    public KeyGenerator getRegisterUsers(int bucket){
        return () -> KeyBuilder.build("game", appConfigFetcher.namespace(), "register", "users", AppContext.get().getAppId(), bucket);
    }
}
