package com.bxm.game.scene.common.core.api.user;

import java.util.Collection;
import java.util.Map;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.bxm.game.scene.common.core.ErrCode;
import com.bxm.game.scene.common.core.bean.GameException;

import lombok.extern.slf4j.Slf4j;

/**
 * 用户映射服务<br/>
 *
 * @author kerry.jiang
 * @date 2021/6/30 14:11
 */
@Slf4j
public abstract class EmptyBxmUserMappingServiceImpl implements BxmUserMappingService {

    private final RestTemplate restTemplate;
    private final String getByUidUrl;
    private final String getByUidsUrl;
    private final String getByCuidUrl;
    private final String getAssetByCuidUrl;
    private final String getAssetLastByIdsUrl;

    public EmptyBxmUserMappingServiceImpl(RestTemplate restTemplate, String apiUrl){
        this.restTemplate = restTemplate;
        this.getByUidUrl = StringUtils.join(apiUrl, "/userMapping/getByUid");
        this.getByUidsUrl = StringUtils.join(apiUrl, "/userMapping/getMappingByUid");
        this.getByCuidUrl = StringUtils.join(apiUrl, "/userMapping/getByCuid");
        this.getAssetByCuidUrl = StringUtils.join(apiUrl, "/userMapping/getAssetByCuid");
        this.getAssetLastByIdsUrl = StringUtils.join(apiUrl, "/userMapping/getAssetLastByIds");
    }

    @Override
    public String getByUid(String appid, String uid, boolean skipCache) {
        ResponseEntity<String> entity;
        try {
            StringBuilder url = new StringBuilder(getByUidUrl);
            url.append("?appId=");
            url.append(appid);
            url.append("&uid=");
            url.append(uid);
            url.append("&skipCache=");
            url.append(skipCache);
            entity = restTemplate.exchange(url.toString(), HttpMethod.GET, null,
                    String.class);
        }catch (ResourceAccessException e){
            log.error("getByUid error: appid={},uid={},skipCache={},e={}",
                    appid, uid, skipCache, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }catch (Exception e){
            log.error("getByUid error: appid={},uid={},skipCache={},e={}",
                    appid, uid, skipCache, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }

        HttpStatus httpStatus = entity.getStatusCode();
        if(HttpStatus.OK == httpStatus){
            return entity.getBody();
        }else{
            log.error("getByUid fail: httpStatus={},appid={},uid={},skipCache={},body={}",
                    httpStatus.value(), appid, uid, skipCache, entity.getBody());
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }
    }

    @Override
    public Map<String, String> getByUids(String appid, Collection<String> uids) {
        ResponseEntity<String> entity;
        try {
            StringBuilder url = new StringBuilder(getByUidsUrl);
            url.append("?appId=");
            url.append(appid);
            uids.forEach(uid -> {
                url.append("&uids=");
                url.append(uid);
            });
            entity = restTemplate.exchange(url.toString(), HttpMethod.GET, null,
                    String.class);
        }catch (ResourceAccessException e){
            log.error("getByUids error: appid={},uids={},e={}",
                    appid, uids, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }catch (Exception e){
            log.error("getByUids error: appid={},uids={},e={}",
                    appid, uids, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }

        HttpStatus httpStatus = entity.getStatusCode();
        if(HttpStatus.OK == httpStatus){
            return JSON.parseObject(entity.getBody(), new TypeReference<Map<String, String>>(){});
        }else{
            log.error("getByUids fail: httpStatus={},appid={},uids={},body={}",
                    httpStatus.value(), appid, uids, entity.getBody());
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }
    }

    @Override
    public String getByCuid(String appid, String cuid, boolean skipCache) {
        ResponseEntity<String> entity;
        try {
            StringBuilder url = new StringBuilder(getByCuidUrl);
            url.append("?appId=");
            url.append(appid);
            url.append("&cuid=");
            url.append(cuid);
            url.append("&skipCache=");
            url.append(skipCache);
            entity = restTemplate.exchange(url.toString(), HttpMethod.GET, null,
                    String.class);
        }catch (ResourceAccessException e){
            log.error("getByCuid error: appid={},cuid={},skipCache={},e={}",
                    appid, cuid, skipCache, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }catch (Exception e){
            log.error("getByCuid error: appid={},cuid={},skipCache={},e={}",
                    appid, cuid, skipCache, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }

        HttpStatus httpStatus = entity.getStatusCode();
        if(HttpStatus.OK == httpStatus){
            return entity.getBody();
        }else{
            log.error("getByCuid fail: httpStatus={},appid={},cuid={},skipCache={},body={}",
                    httpStatus.value(), appid, cuid, skipCache, entity.getBody());
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }
    }

    @Override
    public String getAssetByCuid(String appid, String cuid, boolean skipCache) {
        ResponseEntity<String> entity;
        try {
            StringBuilder url = new StringBuilder(getAssetByCuidUrl);
            url.append("?appId=");
            url.append(appid);
            url.append("&cuid=");
            url.append(cuid);
            url.append("&skipCache=");
            url.append(skipCache);
            entity = restTemplate.exchange(url.toString(), HttpMethod.GET, null,
                    String.class);
        }catch (ResourceAccessException e){
            log.error("getAssetByCuid error: appid={},cuid={},skipCache={},e={}",
                    appid, cuid, skipCache, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }catch (Exception e){
            log.error("getAssetByCuid error: appid={},cuid={},skipCache={},e={}",
                    appid, cuid, skipCache, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }

        HttpStatus httpStatus = entity.getStatusCode();
        if(HttpStatus.OK == httpStatus){
            return entity.getBody();
        }else{
            log.error("getAssetByCuid fail: httpStatus={},appid={},cuid={},skipCache={},body={}",
                    httpStatus.value(), appid, cuid, skipCache, entity.getBody());
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }
    }


    @Override
    public Map<String, String> getAssetLastByIds(String appid, Collection<String> ids) {
        ResponseEntity<String> entity;
        try {
            StringBuilder url = new StringBuilder(getAssetLastByIdsUrl);
            url.append("?appId=");
            url.append(appid);
            ids.forEach(id -> {
                url.append("&ids=");
                url.append(id);
            });
            entity = restTemplate.exchange(url.toString(), HttpMethod.GET, null,
                    String.class);
        }catch (ResourceAccessException e){
            log.error("getAssetLastByIds error: appid={},ids={},e={}",
                    appid, ids, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }catch (Exception e){
            log.error("getAssetLastByIds error: appid={},ids={},e={}",
                    appid, ids, ExceptionUtils.getFullStackTrace(e));
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }

        HttpStatus httpStatus = entity.getStatusCode();
        if(HttpStatus.OK == httpStatus){
            return JSON.parseObject(entity.getBody(), new TypeReference<Map<String, String>>(){});
        }else{
            log.error("getAssetLastByIds fail: httpStatus={},appid={},ids={},body={}",
                    httpStatus.value(), appid, ids, entity.getBody());
            throw new GameException(ErrCode.INTERNAL_SERVER_ERROR);
        }
    }
}
