package com.bxm.game.scene.common.core.prop;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Map;

import lombok.Data;
import lombok.experimental.Accessors;

/**
 * @author allen
 * @date 2020-10-19
 * @since 1.0
 */
@Data
@Accessors(chain = true)
public class AcquiredPropLog {

    private String assetType;

    private String sceneType;
    private String id;
    private String appId;
    private String uid;

    /**
     * 道具数
     */
    private int propNum;
    /**
     * 翻倍数
     */
    private Number multipleNum;
    /**
     * 是否已经翻倍处理了
     */
    private boolean completedMultiple;
    /**
     * 是否已兑换
     */
    private boolean acquired;
    /**
     * 得到道具的行为
     */
    private String act;
    /**
     * 可获取的时间
     */
    private long startAcquireTime;
    /**
     * 过期时间
     */
    private long endAcquireTime;

    /**
     * 来源。当被领取或翻倍后，这里是获取时的记录。
     */
    private AcquiredPropLog source;
    /**
     * 道具的扩展数据
     */
    private Map<String, Object> ext;

    /**
     * 当前时间是否在可领取的有效时间内
     * @return 返回 true 则可领取
     */
    public boolean isBetweenAcquireTime() {
        return LocalDateTime.now().isAfter(getAcquireTimeInMillis()) &&
                (endAcquireTime == 0 || LocalDateTime.now().isBefore(getExpiredTimeInMillis()));
    }

    private LocalDateTime getAcquireTimeInMillis() {
        return of(getStartAcquireTime());
    }

    private LocalDateTime getExpiredTimeInMillis() {
        return of(getEndAcquireTime());
    }

    /**
     * 当前时间是否已经过期，超过了 {@code endAcquireTime}
     * @param endAcquireTime 过期时间，单位：毫秒
     * @return 过期
     */
    public static boolean isExpired(long endAcquireTime) {
        return (endAcquireTime != 0 && LocalDateTime.now().isAfter(of(endAcquireTime)));
    }

    /**
     * 当前时间是否在 {@code startAcquireTime} 与 {@code endAcquireTime} 之间。
     * @param startAcquireTime 开始时间
     * @param endAcquireTime 结束时间
     * @return 处于中间
     */
    public static boolean isBetween(long startAcquireTime, long endAcquireTime) {
        return LocalDateTime.now().isAfter(of(startAcquireTime)) &&
                (endAcquireTime == 0 || LocalDateTime.now().isBefore(of(endAcquireTime)));
    }

    public static LocalDateTime of(long timeInMillis) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timeInMillis), ZoneId.systemDefault());
    }

    public com.bxm.game.scene.common.dal.entity.AcquiredPropLog to() {
        com.bxm.game.scene.common.dal.entity.AcquiredPropLog e = new com.bxm.game.scene.common.dal.entity.AcquiredPropLog();
        e.setId(getId())
                .setAppId(getAppId())
                .setUid(getUid())
                .setAssetType(getAssetType())
                .setSceneType(getSceneType())
                .setPropNum(getPropNum())
                .setMultipleNum(getMultipleNum())
                .setCompletedMultiple(isCompletedMultiple())
                .setAcquired(isAcquired())
                .setAct(getAct())
                .setStartAcquireTime(getStartAcquireTime())
                .setEndAcquireTime(getEndAcquireTime());
        return e;
    }

    public static AcquiredPropLog of(com.bxm.game.scene.common.dal.entity.AcquiredPropLog e) {
        return new AcquiredPropLog()
                .setId(e.getId())
                .setAppId(e.getAppId())
                .setUid(e.getUid())
                .setAssetType(e.getAssetType())
                .setSceneType(e.getSceneType())
                .setPropNum(e.getPropNum())
                .setMultipleNum(e.getMultipleNum())
                .setCompletedMultiple(e.isTypeOfCompletedMultiple())
                .setAcquired(e.isTypeOfAcquired())
                .setAct(e.getAct())
                .setStartAcquireTime(e.getStartAcquireTime())
                .setEndAcquireTime(e.getEndAcquireTime());
    }
}
