package com.bxm.game.scene.common.core.prop.impl;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;

import com.bxm.game.scene.common.core.assets.AssetService;
import com.bxm.game.scene.common.core.prop.Prop;
import com.bxm.game.scene.common.core.prop.PropGrantor;
import com.bxm.game.scene.common.core.prop.event.PropGrantEvent;
import com.bxm.warcar.integration.eventbus.SyncEventPark;
import com.bxm.warcar.utils.JsonHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * 抽象的道具授予者
 *
 * @author allen
 * @date 2020-11-26
 * @since 1.0
 */
@Slf4j
@ConditionalOnMissingBean(PropGrantor.class)
public class DefaultPropGrantorImpl implements PropGrantor {

    private final AssetService assetService;
    private final SyncEventPark syncEventPark;

    public DefaultPropGrantorImpl(AssetService assetService, SyncEventPark syncEventPark) {
        this.assetService = assetService;
        this.syncEventPark = syncEventPark;
    }

    @Override
    public boolean grant(Prop prop, boolean isMultiple) {
        if (!prop.acquired()) {
            return false;
        }
        if (isMultiple && prop.multipleNum().doubleValue() > 0) {
            return this.multiple(prop);
        } else {
            return this.single(prop);
        }
    }

    @Override
    public boolean single(Prop prop) {
        long after;
        try {
            after = assetService.plus(prop.getAssetType(), prop.propNum(), prop.getAct());
        }catch (Exception e){
            log.error("single error: prop={}, e={}",
                    JsonHelper.convert(prop), ExceptionUtils.getStackTrace(e));
            return false;
        }
        syncEventPark.post(new PropGrantEvent(this, PropGrantEvent.TYPE_SINGLE, prop.getAssetType(), after));
        return true;
    }

    @Override
    public boolean multiple(Prop prop) {
        long after;
        try {
            long value = Math.round(prop.propNum() * prop.multipleNum().doubleValue());
            after = assetService.plus(prop.getAssetType(), value, prop.getAct());
        }catch (Exception e){
            log.error("multiple error: prop={}, e={}",
                    JsonHelper.convert(prop), ExceptionUtils.getStackTrace(e));
            return false;
        }
        syncEventPark.post(new PropGrantEvent(this, PropGrantEvent.TYPE_MULTIPLE, prop.getAssetType(), after));
        return true;
    }
}
