package com.bxm.game.scene.common.core.scene;

import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;

import com.bxm.game.scene.common.core.DefaultConsts;
import com.bxm.game.scene.common.core.DefaultRedisKeyType;
import com.bxm.game.scene.common.core.user.DefaultTimeBoundService;

import lombok.extern.slf4j.Slf4j;

/**
 * <p>抽象的每个时效周期内最多可获取的次数的场景实现。</p>
 *
 * <p>默认每个时效周期内可获取一次，该类重写了 {@link #isAbort(SceneRequest, Map)} 方法。</p>
 *
 * <p>你可以通过重写 {@link #getMaximum(R,Map<Object, Object>)} 来设置每个时效周期内最多可获取的次数。</p>
 *
 * @author kerry.jiang
 * @date 2021/3/19 13:06
 */
@Slf4j
public abstract class AbstractDefaultTimeBoundSceneService<R extends SceneRequest, T extends SceneResponse> extends AbstractMaximumTimesOnDailySceneService<R,T> {

    @Autowired(required = false)
    protected DefaultTimeBoundService defaultTimeBoundService;

    @Override
    protected boolean isAbort(R request, Map<Object, Object> attach) {
        attach.put(DefaultConsts.Attach.ROLLBACK_FREQ, true);
        return defaultTimeBoundService.hIncr(DefaultRedisKeyType.FREQ,
                getAbortedKey(request, attach),
                (after) -> after > getMaximum(request, attach));
    }

    @Override
    protected void afterAbort(R request, Map<Object, Object> attach) {
        if(null != attach.get(DefaultConsts.Attach.ROLLBACK_FREQ)){
            defaultTimeBoundService.hIncrBy(DefaultRedisKeyType.FREQ,
                    getAbortedKey(request, attach), -1);
        }
    }
}
