package com.bxm.game.scene.common.core.scene;

import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;

import com.bxm.game.scene.common.core.DefaultConsts;
import com.bxm.game.scene.common.core.DefaultRedisKeyType;
import com.bxm.game.scene.common.core.user.TodayAtomicService;

import lombok.extern.slf4j.Slf4j;

/**
 * <p>抽象的每天最多可获取的次数的场景实现。</p>
 *
 * <p>默认每天可获取一次，该类重写了 {@link #isAbort(SceneRequest, Map)} 方法。</p>
 *
 * <p>你可以通过重写 {@link #getMaximum(R,Map<Object, Object>)} 来设置每天最多可获取的次数。</p>
 *
 * @author allen
 * @date 2020-12-03
 * @since 1.0
 */
@Slf4j
public abstract class AbstractMaximumTimesOnDailySceneService<R extends SceneRequest, T extends SceneResponse> extends AbstractSceneService<R, T> {

    @Autowired
    protected TodayAtomicService todayAtomicService;

    protected String getAbortedKey(R request, Map<Object, Object> attach) {
        return getSceneType();
    }

    /**
     * 包含分支部分的，用作备用
     */
    protected String getAbortedKeyForBranch(String... codes) {
        StringBuilder sb = new StringBuilder(getSceneType());
        for (String code : codes) {
            sb.append('#').append(code);
        }
        return sb.toString();
    }

    /**
     * 返回每天最多可获取的次数
     * @return 每天最多可获取的次数
     */
    protected int getMaximum(R request, Map<Object, Object> attach) {
        return 1;
    }

    @Override
    protected boolean isAbort(R request, Map<Object, Object> attach) {
        attach.put(DefaultConsts.Attach.ROLLBACK_FREQ, true);
        return todayAtomicService.hIncr(DefaultRedisKeyType.FREQ,
                getAbortedKey(request, attach),
                (after) -> after > getMaximum(request, attach));
    }

    @Override
    protected void afterAbort(R request, Map<Object, Object> attach) {
        if(null != attach.get(DefaultConsts.Attach.ROLLBACK_FREQ)){
            todayAtomicService.hIncrBy(DefaultRedisKeyType.FREQ,
                    getAbortedKey(request, attach), -1);
        }
    }
}
