package com.bxm.game.scene.common.core.scene.multiple;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang.StringUtils;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;

import com.bxm.game.scene.common.core.bean.AppContext;
import com.bxm.game.scene.common.core.ErrCode;
import com.bxm.game.scene.common.core.bean.GameException;
import com.bxm.game.scene.common.core.prop.AcquiredPropLog;
import com.bxm.game.scene.common.core.prop.Prop;
import com.bxm.game.scene.common.core.prop.PropBuilder;
import com.bxm.game.scene.common.core.prop.PropService;
import com.bxm.game.scene.common.core.scene.AbstractSceneService;
import com.bxm.game.scene.common.core.scene.AcquiringOrMultiplyPropEvent;
import com.bxm.game.scene.common.core.scene.DefaultSceneType;
import com.bxm.game.scene.common.core.user.TemporaryAtomicService;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * 翻倍场景
 *
 * @author allen
 * @date 2020-11-26
 * @since 1.0
 */
@Slf4j
public class MultipleSceneServiceImpl extends AbstractSceneService<MultipleSceneRequest, MultipleSceneResponse> implements ApplicationListener<ApplicationReadyEvent> {

    private final BigDecimal ONE = new BigDecimal("1");
    private final String ACQUIRED_NUM = "acquiredNum";

    private final Map<String, MultipleConditional> condtionalMap = Maps.newHashMap();
    private final PropService propService;
    private final TemporaryAtomicService temporaryAtomicService;
    private final PropBuilder propBuilder;

    public MultipleSceneServiceImpl(PropService propService, TemporaryAtomicService temporaryAtomicService, PropBuilder propBuilder) {
        this.propService = propService;
        this.temporaryAtomicService = temporaryAtomicService;
        this.propBuilder = propBuilder;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        Map<String, MultipleConditional> beans = event.getApplicationContext().getBeansOfType(MultipleConditional.class);
        for (Map.Entry<String, MultipleConditional> e : beans.entrySet()) {
            MultipleConditional bean = e.getValue();
            String conditionSceneType = bean.getSceneType();
            condtionalMap.put(conditionSceneType, bean);
            log.info("Registered MultipleConditional bean: {}", bean);
        }
    }

    @Override
    public boolean isMultipleGrant() {
        return true;
    }

    @Override
    protected String createId(MultipleSceneRequest request) {
        return request.getId();
    }

    @Override
    protected Prop takeProp(MultipleSceneRequest request, Map<Object, Object> attach) {
        if (Objects.isNull(request)) {
            return null;
        }
        String appId = AppContext.get().getAppId();
        String uid = AppContext.get().getUid();
        String id = request.getId();
        String barrier = getSceneType() + id;
        if (temporaryAtomicService.incrAndAbove(barrier, 1)) {
            throw new GameException(ErrCode.NOT_MEET_ACQUIRING_CONDITION);
        }
        AcquiredPropLog takeLog = propService.getTakeLog(id);
        if (Objects.isNull(takeLog)) {
            return null;
        }
        if (!(StringUtils.equals(appId, takeLog.getAppId())
                && StringUtils.equals(uid, takeLog.getUid())
//                && !takeLog.isAcquired()
                && !takeLog.isCompletedMultiple()
                && takeLog.getMultipleNum().doubleValue() > 1)) {
            return null;
        }
        if (condtionalMap.containsKey(takeLog.getSceneType())
                && !condtionalMap.get(takeLog.getSceneType()).checked(takeLog)) {
            // rollback barrier times
            temporaryAtomicService.incrBy(barrier, -1);
            throw new GameException(ErrCode.NOT_MEET_MULTIPLE_CONDITION);
        }

        boolean acquired = takeLog.isAcquired();
        Number multipleNum = acquired ? new BigDecimal(takeLog.getMultipleNum().toString())
                .subtract(ONE).doubleValue() : takeLog.getMultipleNum();
        if(multipleNum.doubleValue() <= 0){
            return null;
        }
        Prop prop = propBuilder.rebuild(takeLog.getAssetType(), takeLog.getPropNum(),
                multipleNum, takeLog.getAct(), takeLog.getSceneType());
        if (Objects.isNull(prop)) {
            return null;
        }
        propService.saveTakeLog(takeLog.setAcquired(true).setCompletedMultiple(true));

        int acquiredNum;
        if (acquired) {
            acquiredNum = Math.round(prop.propNum() * (prop.multipleNum().floatValue() + 1));
        }else{
            acquiredNum = Math.round(prop.propNum() * prop.multipleNum().floatValue());
        }
        attach.put(ACQUIRED_NUM, acquiredNum);
        return prop;
    }

    @Override
    protected MultipleSceneResponse createResponse(MultipleSceneRequest request, String id, Prop prop, Map<Object, Object> attach) {
        MultipleSceneResponse response = new MultipleSceneResponse();
        response.setAcquiredNum(TypeHelper.castToInt(attach.get(ACQUIRED_NUM)));
        return response;
    }

    @Override
    protected void afterCompletion(MultipleSceneRequest request, MultipleSceneResponse response, Prop prop, Map<Object, Object> attach) {
        getSyncEventPark().post(new MultipleSceneEvent(this, request, response, prop.getSceneType()));
        getSyncEventPark().post(new AcquiringOrMultiplyPropEvent(this, request, response, prop.getSceneType()));
    }

    @Override
    public String getSceneType() {
        return DefaultSceneType.MULTIPLE;
    }

    @Override
    public Class<MultipleSceneRequest> getRequestClass() {
        return MultipleSceneRequest.class;
    }
}
