package com.bxm.game.scene.common.core.scene.signin.continuous;

import java.util.Date;
import java.util.Map;

import com.bxm.game.scene.common.core.DefaultConsts;
import com.bxm.game.scene.common.core.DefaultDataField;
import com.bxm.game.scene.common.core.DefaultRedisKeyType;
import com.bxm.game.scene.common.core.bean.enums.ContinuousSigninRoundTypeEnum;
import com.bxm.game.scene.common.core.prop.Prop;
import com.bxm.game.scene.common.core.scene.DefaultSceneType;
import com.bxm.game.scene.common.core.scene.signin.AbstractSigninSceneServiceImpl;
import com.bxm.game.scene.common.core.scene.signin.SigninSceneRequest;
import com.bxm.game.scene.common.core.scene.signin.SigninSceneResponse;
import com.bxm.game.scene.common.core.user.DefaultTimeBoundService;
import com.bxm.game.scene.common.core.util.CommonHelper;
import com.bxm.warcar.utils.DateHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * 连续签到<br/>
 *
 * @author kerry.jiang
 * @date 2021/10/22 17:17
 */
@Slf4j
public abstract class AbstractContinuousSigninSceneServiceImpl<R extends SigninSceneRequest, T extends SigninSceneResponse>
        extends AbstractSigninSceneServiceImpl<R, T> implements ContinuousSigninService<R, T> {

    protected final ContinuousSigninConfigService signinConfigService;
    protected final DefaultTimeBoundService defaultTimeBoundService;
    protected final int maxDays;
    protected final int ttl;

    public AbstractContinuousSigninSceneServiceImpl(ContinuousSigninConfigService signinConfigService,
                                                    DefaultTimeBoundService defaultTimeBoundService) {
        this.signinConfigService = signinConfigService;
        this.defaultTimeBoundService = defaultTimeBoundService;

        maxDays = signinConfigService.getMaxDays();
        if(maxDays < 1){
            throw new RuntimeException("'maxDays' < 1");
        }
        ttl = DefaultConsts.TTL_DAYS * maxDays;
    }

    @Override
    public int getContinuousSignInTimes() {
        long days = defaultTimeBoundService.hGet(DefaultRedisKeyType.SIGN, getSceneType());
        if(days >= maxDays){
            return todaySignin() ? maxDays : 0;
        }

        Date currentDate = new Date();
        String ymd = DateHelper.convertDateToString(currentDate, DateHelper.PATTERN_STR8);
        String last = defaultTimeBoundService.hGet(DefaultRedisKeyType.SIGN,
                DefaultDataField.SIGNIN_LAST, String.class);
        Date ymdDate = DateHelper.convertStringToDate(ymd, DateHelper.PATTERN_STR8);
        Date lastDate = DateHelper.convertStringToDate(last, DateHelper.PATTERN_STR8);
        if(null == lastDate){
            return 0;
        }else if(CommonHelper.orderDays(lastDate, ymdDate) > 2) {
            // 如果以last为第一天，今天是第三天及以上
            return 0;
        }
        return (int)days;
    }

    @Override
    public int getContinuousSignInRounds() {
        long round = defaultTimeBoundService.hGet(DefaultRedisKeyType.FREQ,
                DefaultDataField.SIGNIN_ROUND);
        if(round < 1){
            return 1;
        }

        Date currentDate = new Date();
        String ymd = DateHelper.convertDateToString(currentDate, DateHelper.PATTERN_STR8);
        String last = defaultTimeBoundService.hGet(DefaultRedisKeyType.SIGN,
                DefaultDataField.SIGNIN_LAST, String.class);
        Date ymdDate = DateHelper.convertStringToDate(ymd, DateHelper.PATTERN_STR8);
        Date lastDate = DateHelper.convertStringToDate(last, DateHelper.PATTERN_STR8);
        if(null == lastDate){
            return getRounds(round, ContinuousSigninRoundTypeEnum.FIRST);
        }else if(CommonHelper.orderDays(lastDate, ymdDate) > 2) {
            // 如果以last为第一天，今天是第三天及以上
            return getRounds(round, ContinuousSigninRoundTypeEnum.BREAK);
        }else{
            long days = defaultTimeBoundService.hGet(DefaultRedisKeyType.SIGN, getSceneType());
            if(days >= maxDays){
                if(!todaySignin()){
                    return getRounds(round, ContinuousSigninRoundTypeEnum.NEXT);
                }
            }
        }
        return (int)round;
    }

    @Override
    public boolean todaySignin() {
        return todayAtomicService.hGet(DefaultRedisKeyType.FREQ, getSceneType()) > 0;
    }

    @Override
    protected boolean isAbort(R request, Map<Object, Object> attach) {
        Date currentDate = new Date();
        String ymd = DateHelper.convertDateToString(currentDate, DateHelper.PATTERN_STR8);
        String last = defaultTimeBoundService.hGet(DefaultRedisKeyType.SIGN,
                DefaultDataField.SIGNIN_LAST, String.class);
        if(null != last && last.compareTo(ymd) == 1){
            // 不支持补签
            return true;
        }
        attach.put(DefaultConsts.Attach.F_1, ymd);
        attach.put(DefaultConsts.Attach.F_2, last);
        return super.isAbort(request, attach);
    }

    @Override
    protected Prop takeProp(R request, Map<Object, Object> attach) {
        String ymd = (String)attach.get(DefaultConsts.Attach.F_1);
        String last = (String)attach.get(DefaultConsts.Attach.F_2);
        Date ymdDate = DateHelper.convertStringToDate(ymd, DateHelper.PATTERN_STR8);
        Date lastDate = DateHelper.convertStringToDate(last, DateHelper.PATTERN_STR8);
        int days;
        if(null == lastDate){
            days = 1;
            defaultTimeBoundService.hSet(DefaultRedisKeyType.SIGN, getSceneType(),
                    String.valueOf(days), ttl);
            addRounds(ContinuousSigninRoundTypeEnum.FIRST);
        }else if(CommonHelper.orderDays(lastDate, ymdDate) > 2){
            // 如果以last为第一天，今天是第三天及以上
            days = 1;
            defaultTimeBoundService.hSet(DefaultRedisKeyType.SIGN, getSceneType(),
                    String.valueOf(days), ttl);
            addRounds(ContinuousSigninRoundTypeEnum.BREAK);
        }else{
            days = (int) defaultTimeBoundService.hIncr(DefaultRedisKeyType.SIGN, getSceneType(), ttl);
            if (days > maxDays) {
                long minus = days - 1;
                // 大于{maxDays},重置为1
                days = 1;
                if(defaultTimeBoundService.hIncrBy(DefaultRedisKeyType.SIGN, getSceneType(), -minus, ttl) < 1){
                    defaultTimeBoundService.hIncrBy(DefaultRedisKeyType.SIGN, getSceneType(), minus, ttl);
                }
                addRounds(ContinuousSigninRoundTypeEnum.NEXT);
            }
        }
        defaultTimeBoundService.hSet(DefaultRedisKeyType.SIGN,
                DefaultDataField.SIGNIN_LAST, ymd, ttl);
        attach.put(DefaultConsts.Attach.F_3, days);
        return signinConfigService.getProps(request.getAcquired(), days);
    }

    /**
     * 增加轮次
     */
    protected void addRounds(ContinuousSigninRoundTypeEnum roundTypeEnum){
        if(ContinuousSigninRoundTypeEnum.FIRST == roundTypeEnum){
            defaultTimeBoundService.hSet(DefaultRedisKeyType.FREQ,
                    DefaultDataField.SIGNIN_ROUND, String.valueOf(1));
        }else if(ContinuousSigninRoundTypeEnum.BREAK == roundTypeEnum){
            defaultTimeBoundService.hIncr(DefaultRedisKeyType.FREQ, DefaultDataField.SIGNIN_ROUND);
        } if(ContinuousSigninRoundTypeEnum.NEXT == roundTypeEnum){
            defaultTimeBoundService.hIncr(DefaultRedisKeyType.FREQ, DefaultDataField.SIGNIN_ROUND);
        }
    }

    /**
     * 获取轮次
     */
    protected int getRounds(long round, ContinuousSigninRoundTypeEnum roundTypeEnum){
        if(ContinuousSigninRoundTypeEnum.FIRST == roundTypeEnum){
            round = 1;
        }else if(ContinuousSigninRoundTypeEnum.BREAK == roundTypeEnum){
            round += 1;
        } if(ContinuousSigninRoundTypeEnum.NEXT == roundTypeEnum){
            round += 1;
        }
        return (int)round;
    }

    @Override
    protected T createResponse(R request, String id, Prop prop, Map<Object, Object> attach) {
        T response = getInstanceResponse(request);
        response.setDays((int)attach.get(DefaultConsts.Attach.F_3));
        return response;
    }

    @Override
    public String getSceneType() {
        return DefaultSceneType.SIGNIN;
    }
}
