package com.bxm.game.scene.common.core.user;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.bxm.game.scene.common.core.DefaultRedisKeyType;
import com.bxm.game.scene.common.core.bean.AppContext;
import com.bxm.game.scene.common.core.bean.enums.RedisStorageEnum;
import com.bxm.warcar.cache.KeyGenerator;

/**
 * 动态时效性<br/>
 *
 * @author kerry.jiang
 * @date 2021/3/17 19:48
 */
public interface AbstractTimeBoundService {

    /**
     * 获取redis key
     *
     * @param type 类目
     * @return
     */
    KeyGenerator getKey(String type);

    /**
     * 不将锚点日期作为key的时效类目
     * @return
     */
    default List<String> ignoreBoundAnchorTypes(){
        return Arrays.asList(DefaultRedisKeyType.ANCHOR);
    }

    /**
     * 获取时效数据的ttl（秒）
     * @return
     */
    default int getTtl(){
        Long l = (AppContext.get().getBoundLimit() - System.currentTimeMillis()) / 1000L;
        return l.intValue();
    }

    /**
     * 获取库中的锚定日期，值为空时保存并使用新的锚定日期(用户设置AppContext中的anchor属性)<br/>
     *
     * @param cycleTimes 周期时长（秒）
     * @param delay 用户数据延迟多久清空（秒）
     * @param force 是否强制更新ttl
     * @param predicate 是否强制更换锚定日期，函数入参为当下的锚定日期
     * @param supplier 新的锚定日期
     * @return 锚定日期
     */
    String saveGetAnchor(int cycleTimes, int delay, boolean force,
                         Predicate<String> predicate, Supplier<String> supplier);

    /**
     * 获取指定 {@code key} 的ttl值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @return redis key实时ttl值，单位：秒
     */
    Long ttl(String type);

    /**
     * 重置ttl
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param ttl ttl值
     */
    void expire(String type, int ttl);

    /**
     * 重置ttl
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     */
    default void expire(String type){
        expire(type, getTtl());
    }

    /**
     * 获取所有的存储数据
     * @return 数据
     */
    Map<String, Object> getAll();

    /**
     * type存储类型映射关系<br/>
     *
     * @return 时效类目 -> 存储类型
     */
    Map<String, RedisStorageEnum> getTypeMap();

    /**
     * 获取指定 {@code key} 的数值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @return 数值
     */
    default long get(String type){
        return get(type, 0L);
    }

    /**
     * 获取指定 {@code key} 的数值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param defaultValue 默认值
     * @return 数值
     */
    long get(String type, long defaultValue);

    /**
     * 获取指定 {@code key} 的值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param cls 值类型
     * @return 值
     */
    <T> T get(String type, Class<T> cls);

    /**
     * 保存值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param value 值
     * @param ttl 有效时间（秒）
     */
    void set(String type, String value, int ttl);

    /**
     * 在当前 {@code key} 对应的数值 +1
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param ttl 有效时间（秒）
     * @return 增加后的数值
     */
    default long incr(String type, int ttl) {
        return incrBy(type, 1, ttl);
    }

    /**
     * 在当前 {@code key} 对应的数值增量 {@code value}
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param value value
     * @param ttl 有效时间（秒）
     * @return 增加后的数值
     */
    long incrBy(String type, long value, int ttl);

    /**
     * 在当前 {@code key} 对应的数值 +1，并判断返回的值是否大于 {@code above}
     *
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param above 比较值
     * @param ttl 有效时间（秒）
     * @return 判断结果
     */
    default boolean incrAndAbove(String type, long above, int ttl) {
        return incr(type, (after) -> after > above, ttl);
    }

    /**
     * 在当前 {@code key} 对应的数值 +1，并返回 {@code predicate} 判断后的结果
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param predicate 判断接口，参数值为增加后的数值
     * @param ttl 有效时间（秒）
     * @return 判断结果
     */
    default boolean incr(String type, Predicate<Long> predicate, int ttl) {
        return incrBy(type, 1, predicate, ttl);
    }

    /**
     * 在当前 {@code key} 对应的数值增量 {@code value}，并返回 {@code predicate} 判断后的结果
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param value value
     * @param predicate 判断接口，参数值为增加后的数值
     * @param ttl 有效时间（秒）
     * @return 判断结果
     */
    boolean incrBy(String type, long value, Predicate<Long> predicate, int ttl);

    /**
     * 删除某个key
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     */
    void del(String type);

    /**
     * 获取指定 {@code key} 的数值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @return 数值
     */
    default long hGet(String type, String field){
        return hGet(type, field, 0L);
    }

    /**
     * 获取指定 {@code key} 的数值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param defaultValue 默认值
     * @return 数值
     */
    long hGet(String type, String field, long defaultValue);

    /**
     * 获取指定 {@code key} 的值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param cls 值类型
     * @return 值
     */
    <T> T hGet(String type, String field, Class<T> cls);

    /**
     * 获取指定 {@code key} 的数值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @return 数值
     */
    Map<String, Object> hGetAll(String type);

    /**
     * 获取指定 {@code key} 的数值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @return 数值
     */
    <T> Map<String, T> hGetAll(String type, Class<T> cls);

    /**
     * 保存值
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param value 值
     * @param ttl 有效时间（秒）
     */
    void hSet(String type, String field, String value, int ttl);

    /**
     * 在当前 {@code key} 对应的数值 +1
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param ttl 有效时间（秒）
     * @return 增加后的数值
     */
    default long hIncr(String type, String field, int ttl) {
        return hIncrBy(type, field, 1, ttl);
    }

    /**
     * 在当前 {@code key} 对应的数值增量 {@code value}
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param value value
     * @param ttl 有效时间（秒）
     * @return 增加后的数值
     */
    long hIncrBy(String type, String field, long value, int ttl);

    /**
     * 在当前 {@code key} 对应的数值 +1，并判断返回的值是否大于 {@code above}
     *
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param above 比较值
     * @param ttl 有效时间（秒）
     * @return 判断结果
     */
    default boolean hIncrAndAbove(String type, String field, long above, int ttl) {
        return hIncr(type, field, (after) -> after > above, ttl);
    }

    /**
     * 在当前 {@code key} 对应的数值 +1，并返回 {@code predicate} 判断后的结果
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param predicate 判断接口，参数值为增加后的数值
     * @param ttl 有效时间（秒）
     * @return 判断结果
     */
    default boolean hIncr(String type, String field, Predicate<Long> predicate, int ttl) {
        return hIncrBy(type, field, 1, predicate, ttl);
    }

    /**
     * 在当前 {@code key} 对应的数值增量 {@code value}，并返回 {@code predicate} 判断后的结果
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     * @param value value
     * @param predicate 判断接口，参数值为增加后的数值
     * @param ttl 有效时间（秒）
     * @return 判断结果
     */
    boolean hIncrBy(String type, String field, long value, Predicate<Long> predicate, int ttl);

    /**
     * 删除某些field
     *
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param field field
     */
    void hDel(String type, String... field);

    /**
     * 二进制偏移标识<br/>
     *
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param offset 从左往右的偏移量
     * @param value 偏移的值是否改成1，true:改为1，false:改为0
     */
    default void setbit(String type, long offset, boolean value){
        setbit(type, offset, value, getTtl());
    }

    /**
     * 二进制偏移标识<br/>
     *
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @param offset 从左往右的偏移量
     * @param value 偏移的值是否改成1，true:改为1，false:改为0
     * @param ttl 有效时间（秒）
     */
    void setbit(String type, long offset, boolean value, int ttl);

    /**
     * 获取二进制位上带“1”值的数量<br/>
     *
     * @param type 时效类目，用于{@code key.getTimeBound(type)}
     * @return
     */
    long bitcount(String type);
}