package com.bxm.game.scene.common.core.user.redis;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Qualifier;

import com.bxm.game.scene.common.core.AppConfigFetcher;
import com.bxm.game.scene.common.core.Key;
import com.bxm.game.scene.common.core.bean.enums.RedisStorageEnum;
import com.bxm.game.scene.common.core.user.BaseAtomicService;
import com.bxm.game.scene.common.core.util.CommonHelper;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * 原子数据<br/>
 *
 * @author kerry.jiang
 * @date 2021/10/22 9:49
 */
@Slf4j
public abstract class BaseAtomicServiceImpl implements BaseAtomicService {

    /**
     * 时效类目 -> 存储类型
     */
    protected final Map<String, RedisStorageEnum> TYPE_MAP = new HashMap<>();

    protected final AppConfigFetcher appConfigFetcher;
    protected final Key key;
    protected final Fetcher fetcher;
    protected final Counter counter;
    protected final Updater updater;

    public BaseAtomicServiceImpl(AppConfigFetcher appConfigFetcher, Key key,
                                 @Qualifier("jedisFetcher") Fetcher fetcher,
                                 @Qualifier("jedisUpdater") Updater updater,
                                 @Qualifier("jedisCounter") Counter counter) {
        this.appConfigFetcher = appConfigFetcher;
        this.key = key;
        this.fetcher = fetcher;
        this.counter = counter;
        this.updater = updater;

        Map<String, RedisStorageEnum> m = getTypeMap();
        if(null != m){
            TYPE_MAP.putAll(m);
        }
    }

    @Override
    public Long ttl(String type) {
        JedisPool jedisPool = CommonHelper.getJedisPool(fetcher);
        try (Jedis jedis = jedisPool.getResource()){
            return jedis.ttl(getKey(type).generateKey());
        }
    }

    @Override
    public void expire(String type, int ttl) {
        updater.expire(getKey(type), ttl);
    }

    @Override
    public Map<String, Object> getAll() {
        Map<String, Object> map = new HashMap();
        Set<String> types = TYPE_MAP.keySet();
        for (String type : types) {
            RedisStorageEnum storageEnum = TYPE_MAP.get(type);

            Object obj = null;
            if(RedisStorageEnum.HASH == storageEnum){
                obj = fetcher.hfetchall(getKey(type), Object.class);
            }else{
                obj = fetcher.fetch(getKey(type), Object.class);
            }
            map.put(type, obj);
        }
        return map;
    }

    @Override
    public Map<String, RedisStorageEnum> getTypeMap() {
        Map<String, RedisStorageEnum> map = new HashMap<>();
        return map;
    }

    @Override
    public long get(String type, long defaultValue) {
        return Optional.ofNullable(counter.get(getKey(type))).orElse(defaultValue);
    }

    @Override
    public <T> T get(String type, Class<T> cls) {
        return fetcher.fetch(getKey(type), cls);
    }

    @Override
    public void set(String type, String value, int ttl) {
        updater.update(getKey(type), value, ttl);
    }

    @Override
    public long incrBy(String type, long value, int ttl) {
        return Optional.ofNullable(counter.incrementByAndGet(getKey(type), value, ttl)).orElse(0L);
    }

    @Override
    public boolean incrBy(String type, long value, Predicate<Long> predicate, int ttl) {
        long l = this.incrBy(type, value);
        return predicate.test(l);
    }

    @Override
    public void del(String type) {
        updater.remove(getKey(type));
    }

    @Override
    public long hGet(String type, String field, long defaultValue) {
        return Optional.ofNullable(counter.hget(getKey(type), field)).orElse(defaultValue);
    }

    @Override
    public <T> T hGet(String type, String field, Class<T> cls) {
        return fetcher.hfetch(getKey(type), field, cls);
    }

    @Override
    public Map<String, Object> hGetAll(String type) {
        return fetcher.hfetchall(getKey(type), Object.class);
    }

    @Override
    public <T> Map<String, T> hGetAll(String type, Class<T> cls) {
        return fetcher.hfetchall(getKey(type), cls);
    }

    @Override
    public void hSet(String type, String field, String value, int ttl) {
        updater.hupdate(getKey(type), field, value, ttl);
    }

    @Override
    public long hIncrBy(String type, String field, long value, int ttl) {
        return Optional.ofNullable(counter.hincrementByAndGet(getKey(type), field, value, ttl)).orElse(0L);
    }

    @Override
    public boolean hIncrBy(String type, String field, long value, Predicate<Long> predicate, int ttl) {
        long l = this.hIncrBy(type, field, value, ttl);
        return predicate.test(l);
    }

    @Override
    public void hDel(String type, String... field) {
        updater.hremove(getKey(type), field);
    }
}
