package com.bxm.game.scene.common.core.web.controllers;

import java.util.Map;
import java.util.Objects;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.game.scene.common.core.AppConfigFetcher;
import com.bxm.game.scene.common.core.fun.archives.CustomArchiveHandler;
import com.bxm.game.scene.common.core.fun.archives.CustomArchiveHandlerFactory;
import com.bxm.game.scene.common.core.assets.AssetService;
import com.bxm.game.scene.common.core.bean.enums.DataTypeEnum;
import com.bxm.game.scene.common.core.bean.vo.ArchiveVo;
import com.bxm.game.scene.common.core.scene.signin.continuous.ContinuousSigninService;
import com.bxm.game.scene.common.core.transfer.AssetArchiveTransfer;
import com.bxm.game.scene.common.core.transfer.DataTransfer;
import com.bxm.game.scene.common.core.user.DefaultTimeBoundService;
import com.bxm.game.scene.common.core.user.PermanentAtomicService;
import com.bxm.game.scene.common.core.user.TodayAtomicService;
import com.bxm.game.scene.common.core.user.UserActiveService;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @author allen
 * @date 2020-11-30
 * @since 1.0
 */
@Slf4j
@RestController
@RequestMapping("/archive")
public class ArchiveController {

    private final AppConfigFetcher appConfigFetcher;
    private final CustomArchiveHandlerFactory customArchiveHandlerFactory;
    private final DataTransfer dataTransfer;
    private final AssetArchiveTransfer assetArchiveTransfer;
    private final TodayAtomicService todayAtomicService;
    private final PermanentAtomicService permanentAtomicService;
    private final DefaultTimeBoundService defaultTimeBoundService;
    private final AssetService assetService;
    private final UserActiveService userActiveService;

    @Autowired(required = false)
    private ContinuousSigninService continuousSigninService;

    public ArchiveController(AppConfigFetcher appConfigFetcher, CustomArchiveHandlerFactory customArchiveHandlerFactory,
                             DataTransfer dataTransfer, AssetArchiveTransfer assetArchiveTransfer, TodayAtomicService todayAtomicService,
                             PermanentAtomicService permanentAtomicService, DefaultTimeBoundService defaultTimeBoundService,
                             AssetService assetService, UserActiveService userActiveService) {
        this.appConfigFetcher = appConfigFetcher;
        this.customArchiveHandlerFactory = customArchiveHandlerFactory;
        this.dataTransfer = dataTransfer;
        this.assetArchiveTransfer = assetArchiveTransfer;
        this.todayAtomicService = todayAtomicService;
        this.permanentAtomicService = permanentAtomicService;
        this.defaultTimeBoundService = defaultTimeBoundService;
        this.assetService = assetService;
        this.userActiveService = userActiveService;
    }

    @GetMapping("/fetch")
    public ResponseEntity<ArchiveVo> fetch() {
        Map<String, Object> custom = Maps.newHashMap();

        ImmutableSet<CustomArchiveHandler> handlers = customArchiveHandlerFactory.getHandlers();
        for (CustomArchiveHandler handler : handlers) {
            String displayFieldName = handler.getDisplayFieldName();
            if (custom.containsKey(displayFieldName)) {
                log.warn("duplicated key: {}", displayFieldName);
            }
            custom.put(displayFieldName, handler.getArchiveObjects());
        }

        ArchiveVo.ArchiveVoBuilder builder = ArchiveVo.builder()
                .today(dataTransfer.transfer(DataTypeEnum.TODAY,todayAtomicService.getAll()))
                .permanent(dataTransfer.transfer(DataTypeEnum.PERMANENT,permanentAtomicService.getAll()))
                .bound(dataTransfer.transfer(DataTypeEnum.BOUND, defaultTimeBoundService.getAll()))
                .assets(assetArchiveTransfer.transfer(assetService.getAll()))
                .custom(custom)
                .time(System.currentTimeMillis());

        if (Objects.nonNull(continuousSigninService)) {
            builder.continuousSignInDays(continuousSigninService.getContinuousSignInTimes());
        }
        if (appConfigFetcher.activatingOnFetchArchive()) {
            userActiveService.active();
        }
        if (appConfigFetcher.showUserActiveDays()) {
            builder.activeDays(userActiveService.getActiveDays());
        }

        return ResponseEntity.ok(builder.build());
    }
}
