package com.bxm.game.scene.common.core.web.controllers;

import javax.servlet.http.HttpServletRequest;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import com.bxm.game.scene.common.core.AppConfigFetcher;
import com.bxm.game.scene.common.core.bean.GameException;
import com.bxm.game.scene.common.core.fun.action.SceneActionFetcher;
import com.bxm.game.scene.common.core.fun.action.SceneActionRequest;
import com.bxm.game.scene.common.core.fun.action.SceneActionResponse;
import com.bxm.game.scene.common.core.fun.action.SceneActionServiceFactory;
import com.bxm.game.scene.common.core.scene.SceneRequest;
import com.bxm.game.scene.common.core.scene.SceneResponse;
import com.bxm.game.scene.common.core.scene.SceneService;
import com.bxm.game.scene.common.core.scene.SceneServiceFactory;

import lombok.extern.slf4j.Slf4j;

/**
 * @author allen
 * @date 2020-10-16
 * @since 1.0
 */
@Slf4j
@RestController
@RequestMapping("/scene")
public class SceneController {

    private final SceneServiceFactory factory;
    private final SceneActionServiceFactory actionFactory;
    private final AppConfigFetcher appConfigFetcher;

    public SceneController(SceneServiceFactory factory, SceneActionServiceFactory actionFactory,
                           AppConfigFetcher appConfigFetcher) {
        this.factory = factory;
        this.actionFactory = actionFactory;
        this.appConfigFetcher = appConfigFetcher;
    }

    /**
     * 获取道具
     *
     * @param sceneType 场景
     * @param message 场景所需要的请求参数
     * @return 获取结果
     */
    @PostMapping("/take/{sceneType}")
    public ResponseEntity<SceneResponse> take(@PathVariable("sceneType") String sceneType,
                                              @RequestBody(required = false) String message) {
        SceneService service = factory.get(sceneType);
        if (Objects.isNull(service)) {
            return ResponseEntity.badRequest().build();
        }
        SceneRequest request = service.convert(message);
        SceneResponse response = service.take(request);
        return ResponseEntity.ok(response);
    }

    /**
     * 批量获取道具
     *
     * @param sceneType 场景
     * @param message 场景所需要的请求参数
     * @return 获取结果
     */
    @PostMapping("/takes/{sceneType}")
    public ResponseEntity<List<SceneResponse>> takes(@PathVariable("sceneType") String sceneType,
                                                @RequestBody(required = false) String message,
                                                HttpServletRequest httpRequest) {
        SceneService service = factory.get(sceneType);
        if (Objects.isNull(service)) {
            return ResponseEntity.badRequest().build();
        }
        List<SceneRequest> requests = service.convertList(message);
        if(CollectionUtils.isEmpty(requests)){
            return ResponseEntity.badRequest().build();
        }
        List<SceneResponse> reps = new ArrayList<>();
        try {
            for (SceneRequest request : requests) {
                SceneResponse response = service.take(request);
                reps.add(response);
            }
        }catch (GameException e){
//            if(appConfigFetcher.debugIps().contains(IpHelper.getIpFromHeader(httpRequest))){
//                log.warn("takes warn: sceneType={},rqSize={},rsSize={},{}",
//                        sceneType, requests.size(), reps.size(), e.getErrCode());
//            }
            log.warn("takes warn: sceneType={},rqSize={},rsSize={},{}",
                        sceneType, requests.size(), reps.size(), e.getErrCode());
        }
        return ResponseEntity.ok(reps);
    }

    /**
     * 执行
     *
     * @param sceneType 场景
     * @param message 场景所需要的请求参数
     * @return 执行结果
     */
    @PostMapping("/action/{sceneType}")
    public ResponseEntity<SceneActionResponse> action(@PathVariable("sceneType") String sceneType,
                                              @RequestBody(required = false) String message) {
        SceneActionFetcher service = actionFactory.get(sceneType);
        if (Objects.isNull(service)) {
            return ResponseEntity.badRequest().build();
        }
        SceneActionRequest request = service.convertAction(message);
        SceneActionResponse response = service.action(request);
        return ResponseEntity.ok(response);
    }
}
