package com.bxm.game.scene.common.core.web.interceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.web.servlet.HandlerInterceptor;

import com.bxm.game.scene.common.core.AppConfigFetcher;
import com.bxm.game.scene.common.core.bean.AppContext;
import com.bxm.game.scene.common.core.fun.header.RequestHeaderHandler;
import com.bxm.game.scene.common.core.user.UserRegisterService;
import com.bxm.game.scene.common.core.user.dao.UserDao;
import com.bxm.warcar.integration.dc.dot.Device;
import com.bxm.warcar.utils.IpHelper;
import com.bxm.warcar.utils.JsonHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * @author allen
 * @date 2020-10-14
 * @since 1.0
 */
@Slf4j
public class AppContextInterceptor implements HandlerInterceptor {

    private final String UNDEFINED = "undefined";

    private final String X_GCODE = "x-gcode";
    private final String X_APPID = "x-appid";
    private final String X_DEVICE = "x-device";
    private final String X_DID = "x-did";

    private final UserDao userDao;
    private final RequestHeaderHandler requestHeaderHandler;
    private final UserRegisterService userRegisterService;
    private final AppConfigFetcher appConfigFetcher;

    public AppContextInterceptor(UserDao userDao, RequestHeaderHandler requestHeaderHandler,
                                 UserRegisterService userRegisterService, AppConfigFetcher appConfigFetcher){
        this.userDao = userDao;
        this.requestHeaderHandler = requestHeaderHandler;
        this.userRegisterService = userRegisterService;
        this.appConfigFetcher = appConfigFetcher;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        String gcode = request.getHeader(X_GCODE);
        String appid = request.getHeader(X_APPID);
        String device = request.getHeader(X_DEVICE);
        String did = request.getHeader(X_DID);
        // 校验参数
        if (checkHeadParam(response, gcode, appid, device)){
            return false;
        }

        AppContext context = AppContext.getInstance();
        context.setGameCode(gcode).setAppId(appid).setXDevice(device).setDid(did);

        if (StringUtils.isNotBlank(did)) {
            Device deviceObject = Device.decrypt(did);
            context.setDevice(deviceObject);
            String newDevice = requestHeaderHandler.fetchDeviceValue(deviceObject);
            context.setDidDevice(newDevice);

            // UID 之前使用的是明文 imei / android id，国家规定不能非法收集用户隐私，IMEI 属于。
            // 所以在这之后 Android端 UID使用不可逆密文，为了兼容用户的历史数据，需要根据 did 解密反查出当时的UID值。
            // modified by Allen
            if (requestHeaderHandler.isOverrideDeviceValue(appid, device, deviceObject)) {
                if (!StringUtils.equals(newDevice, device)) {
                    if (log.isDebugEnabled()) {
                        log.debug("{} Not equals, device is:{}, except: {}, but: {}", appid, JsonHelper.convert(deviceObject), device, newDevice);
                    }
                }
                // 从Device里获取原来的 UID
                if (StringUtils.isNotBlank(newDevice)) {
                    device = newDevice;
                }
            }
        }

        context.setDevUid(device).setUid(device);

        boolean ret = userDao.handleAppContext(request, response, context);
        if(ret){
            // 注册用户信息
            userRegisterService.register();
        }
        if(appConfigFetcher.debugIps().contains(IpHelper.getIpFromHeader(request))){
            log.info("AppContext={}", JsonHelper.convert(context));
        }
        return ret;
    }

    /**
     * 校验参数
     */
    private boolean checkHeadParam(HttpServletResponse response, String gcode, String appid, String device) {
        if (StringUtils.isBlank(gcode) || UNDEFINED.equals(gcode)) {
            log.info("Required header name: 'gcode' is blank or undefined!");
            response.setStatus(HttpStatus.ACCEPTED.value());
            return true;
        }
        if (StringUtils.isBlank(appid) || UNDEFINED.equals(appid)) {
            log.info("Required header name: 'appid' is blank or undefined!");
            response.setStatus(HttpStatus.ACCEPTED.value());
            return true;
        }
        if (StringUtils.isBlank(device) || UNDEFINED.equals(device)) {
            log.info("Required header name: 'device' is blank or undefined!");
            response.setStatus(HttpStatus.ACCEPTED.value());
            return true;
        }
        return false;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        AppContext.remove();
    }
}
