package com.bxm.game.scene.common.core.withdraw;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bxm.game.scene.common.core.AppConfigFetcher;
import com.bxm.game.scene.common.core.ErrCode;
import com.bxm.game.scene.common.core.bean.AppContext;
import com.bxm.game.scene.common.core.bean.GameException;
import com.bxm.game.scene.common.core.withdraw.dao.WithdrawDao;
import com.bxm.game.scene.common.dal.entity.PrizeLog;
import com.bxm.game.scene.common.dal.service.IPrizeLogService;
import com.bxm.warcar.id.IdGenerator;

import lombok.extern.slf4j.Slf4j;

/**
 * 提现服务-网络版<br/>
 *
 * @author kerry.jiang
 * @date 2021/3/23 17:15
 */
@Slf4j
@ConditionalOnBean(WithdrawDao.class)
public class WithdrawServiceImpl implements WithdrawService {

    /**
     * 待发放
     */
    private final int STATUS_SUBMIT = 1;

    private final IPrizeLogService iPrizeLogService;
    private final AppConfigFetcher appConfigFetcher;
    private final IdGenerator id;
    private final WithdrawDao withdrawDao;

    public WithdrawServiceImpl(IPrizeLogService iPrizeLogService, AppConfigFetcher appConfigFetcher,
                               IdGenerator id, WithdrawDao withdrawDao) {
        this.iPrizeLogService = iPrizeLogService;
        this.appConfigFetcher = appConfigFetcher;
        this.id = id;
        this.withdrawDao = withdrawDao;
    }

    @Override
    public void withdraw(WithdrawRequest request) {
        Map<Object, Object> attach = new HashMap<>();
        // 校验提现条件
        if(!withdrawDao.check(request, attach)){
            return ;
        }

        // 执行扣款
        withdrawDao.deduct(request, attach);

        // 提现金额转元
        Double yuan = withdrawDao.toYuan(request, attach);
        BigDecimal bd = new BigDecimal(String.valueOf(yuan));
        Double fen = yuan * 100;

        AppContext ctx = AppContext.get();
        PrizeLog prizeLog = new PrizeLog()
                .setOrderId(id.next())
                .setGameCode(ctx.getGameCode())
                .setAppId(ctx.getAppId())
                .setUid(ctx.getUid())
                .setAppUid(Optional.ofNullable(ctx.getCuid()).orElse(StringUtils.EMPTY))
                .setStatus(STATUS_SUBMIT)
                .setPrizeId(0L)
                .setPrizeName(bd.setScale(2, RoundingMode.HALF_UP).toString())
                .setPrizeNum(1)
                .setName(request.getName())
                .setMobile(request.getMobile())
                .setAddress(StringUtils.EMPTY)
                .setActivityType(appConfigFetcher.activityType())
                .setPayType(request.getPayType())
                .setAccount(request.getAccount())
                .setAmount(fen.intValue());

        // 提现记录保存前,处理提现记录
        withdrawDao.handlePrizeLog(prizeLog, attach);

        try {
            if (!iPrizeLogService.save(prizeLog)) {
                throw new GameException(ErrCode.WITHDRAW_ERR);
            }
            log.info("PrizeLog: {}", prizeLog);
        } catch (Exception e) {
            log.error("save: ", e);
            // 回滚扣款
            withdrawDao.rollback(request, attach);
            throw new GameException(ErrCode.WITHDRAW_ERR);
        }

        // 提现成功后的通知
        withdrawDao.notifySuccess(request, attach, prizeLog);
    }

    @Override
    public List<WithdrawResponse> list() {
        QueryWrapper<PrizeLog> query = withdrawDao.getQuery();
        if(null == query){
            AppContext ctx = AppContext.get();
            query = Wrappers.query(new PrizeLog()
                    .setGameCode(ctx.getGameCode())
                    .setAppId(ctx.getAppId())
                    .setUid(ctx.getUid())
                    .setActivityType(appConfigFetcher.activityType())
            );
            // amount > 0
            query.gt(PrizeLog.AMOUNT, 0);
        }
        return withdrawDao.getResponse(iPrizeLogService.list(query));
    }
}
