package com.bxm.manage.gateway.filter;

import com.bxm.component.jwt.bo.JwtTokenBO;
import com.bxm.component.jwt.util.JwtUtil;
import com.bxm.manage.gateway.properties.GatewayProperties;
import com.bxm.newidea.component.util.WebUtils;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.util.AntPathMatcher;

import javax.servlet.http.HttpServletRequest;

import static com.bxm.manage.gateway.constant.GatewayConstant.*;
import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * 权限前置过滤器，优先级较低，先对请求的合法性进行校验
 *
 * @author liujia
 */
@Slf4j
public class AuthenticationPreFilter extends AbstractZuulFilter {

    private GatewayProperties gatewayProperties;

    private AntPathMatcher antPathMatcher;

    public AuthenticationPreFilter(GatewayProperties gatewayProperties) {
        this.gatewayProperties = gatewayProperties;
        this.antPathMatcher = new AntPathMatcher();
    }

    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    @Override
    public Object run() throws ZuulException {
        try {
            RequestContext requestContext = RequestContext.getCurrentContext();

            HttpServletRequest request = requestContext.getRequest();
            String uri = request.getRequestURI();

            // 判断请求路径是否需要鉴权
            if (!gatewayProperties.isEnableAuth() || isSkipAuthUri(uri)) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}]不需要鉴权即可访问：", uri);
                }

                accessLog(request, null);
                return null;
            }
            String token = obtainToken(request);

            String userId = null;
            // 判断是否存在token
            if (isNotBlank(token)) {
                userId = parseToken(token);
                requestContext.set(USER_ID_KEY, userId);
            }

            accessLog(request, userId);

            // 能解析出userId就认为是正确的token
            if (isNotBlank(userId)) {
                return null;
            }

            log.error("token error,uri:{},ip:{},agent:{},token:{}",
                    request.getRequestURI(),
                    WebUtils.getIpAddr(request),
                    request.getHeader("User-Agent"), token);

            throw new ZuulException("登录凭证无效，请重新登录", HttpStatus.UNAUTHORIZED.value(), "登录凭证无效，请重新登录");
        } catch (ZuulException e) {
            throw e;
        } catch (Exception e) {
            log.error("处理请求失败: {}", RequestContext.getCurrentContext().getRequest().getRequestURI(), e);
            throw new ZuulException("请求处理失败", HttpStatus.BAD_REQUEST.value(), "请求处理失败");
        }
    }

    private String parseToken(String token) {
        JwtTokenBO jwtTokenBO = JwtUtil.parseToken(token, gatewayProperties.getTokenSecret());
        return jwtTokenBO.getBodyWithString(USER_ID_KEY);
    }

    private void accessLog(HttpServletRequest request, String userId) {
        String requestMethod = request.getMethod();

        if (HttpMethod.OPTIONS.matches(requestMethod)) {
            return;
        }

        String requestIp = WebUtils.getIpAddr(request);
        String url = request.getRequestURI();
        String params = getReqeustParam(request);

        //slb负载均衡会在额定的时间间隔内请求,过滤日志
        if (antPathMatcher.match("/app/heartbeat", url)) {
            return;
        }

        log.info(ACCESS, "url:[{}],method:[{}],userId:[{}],requestIp:[{}],params:[{}]",
                url,
                requestMethod,
                userId,
                requestIp,
                params);
    }

    /**
     * 从请求中获取token
     *
     * @param request 请求信息
     * @return 用户tokne，可能为空
     */
    private String obtainToken(HttpServletRequest request) {
        return request.getHeader(TOKEN_HEADER);
    }

    /**
     * 默认对所有请求地址进行鉴权，如果配置了跳过鉴权，则不做处理
     *
     * @param uri 请求地址
     * @return true表示需要进行鉴权
     */
    private boolean isSkipAuthUri(String uri) {
        return gatewayProperties.getSkipAuthUrl()
                .values()
                .stream()
                .anyMatch(skipAuthUrl -> antPathMatcher.match(skipAuthUrl, uri));
    }
}
