package com.bxm.manage.gateway.filter;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.JSONObject;
import com.bxm.newidea.component.util.WebUtils;
import com.google.common.collect.Maps;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.http.HttpServletRequestWrapper;
import com.netflix.zuul.util.HTTPRequestUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.entity.ContentType;
import org.springframework.cloud.netflix.ribbon.support.ResettableServletInputStreamWrapper;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import static com.bxm.manage.gateway.constant.GatewayConstant.CURRENT_USER_ID;
import static com.bxm.manage.gateway.constant.GatewayConstant.USER_ID_KEY;
import static com.bxm.newidea.component.JSON.parseObject;
import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * 请求重构过滤器
 * 设置请求来源系统和当前运营操作人员ID
 *
 * @author liujia
 * @date 2020/05/03 11:50
 */
@Slf4j
public class RefactorFilter extends AbstractZuulFilter {


    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    @Override
    public Object run() {
        RequestContext requestContext = RequestContext.getCurrentContext();
        HttpServletRequest request = requestContext.getRequest();

        setExtendParam(request);

        return null;
    }

    private void setExtendParam(HttpServletRequest request) {
        // 如果没有请求参数 则会返回空 需要初始化
        Map<String, List<String>> queryParams = HTTPRequestUtils.getInstance().getQueryParams();
        if (Objects.isNull(queryParams)) {
            queryParams = Maps.newHashMap();
            RequestContext.getCurrentContext().setRequestQueryParams(queryParams);
        }

        Object currentUserId = RequestContext.getCurrentContext().get(USER_ID_KEY);
        if (currentUserId == null) {
            currentUserId = "0";
        }

        // 在请求Url后直接追加参数
        queryParams.put(CURRENT_USER_ID, Collections.singletonList(currentUserId.toString()));

        RequestContext.getCurrentContext().addZuulRequestHeader(CURRENT_USER_ID, currentUserId.toString());

        if (RequestMethod.POST.name().equalsIgnoreCase(request.getMethod())) {
            addExtendPostParam(request, currentUserId);
        }
    }

    private void addExtendPostParam(HttpServletRequest request, Object currentUserId) {
        if (Objects.equals(ContentType.APPLICATION_JSON.getMimeType(), request.getContentType())) {
            // 对于application/json的请求 还需要在request body中添加字段
            String body = WebUtils.getRequestBodyContent(request);
            JSONObject jsonObject;

            if (isNotBlank(body)) {
                try {
                    jsonObject = parseObject(body);
                } catch (Exception e) {
                    log.info("请求{}的body格式不是一个json结构", request.getRequestURI());
                    jsonObject = new JSONObject();
                }
            } else {
                // 考虑没有携带参数的情况...
                jsonObject = new JSONObject();
            }

            // 添加新字段 srcApp
            jsonObject.put(CURRENT_USER_ID, currentUserId.toString());
            // 包装request 设置进新添加的的请求参数
            RequestContext.getCurrentContext().setRequest(new PostParamChangeRequestWrapper(request, JSON.toJSONBytes(jsonObject)));
        } else {
            if (log.isDebugEnabled()) {
                log.debug("{} 是一个Post请求，但ContentType不等于：{},实际请求contentType为：{}",
                        request.getRequestURI(),
                        ContentType.APPLICATION_JSON.getMimeType(),
                        request.getContentType());
            }
        }
    }

    /**
     * post 参数修改的包装类
     *
     * @author Gonzo
     * @date 2020-05-15 15:14
     * @since 3.2.0
     */
    private class PostParamChangeRequestWrapper extends HttpServletRequestWrapper {

        private byte[] contentData;

        private HttpServletRequest req;

        PostParamChangeRequestWrapper(HttpServletRequest req, byte[] contentData) {
            super(req);
            this.req = req;
            this.contentData = contentData;
        }

        @Override
        public byte[] getContentData() {
            return this.contentData;
        }

        @Override
        public int getContentLength() {
            return this.contentData.length;
        }

        @Override
        public long getContentLengthLong() {
            return this.contentData.length;

        }

        @Override
        public ServletInputStream getInputStream() throws IOException {
            if (Objects.nonNull(this.contentData)) {
                return new ResettableServletInputStreamWrapper(this.contentData);
            }
            return super.getInputStream();
        }

        @Override
        public BufferedReader getReader() throws IOException {
            if (Objects.nonNull(this.contentData)) {
                String enc = Objects.isNull(req.getCharacterEncoding()) ? "UTF-8" : req.getCharacterEncoding();
                byte[] data = this.contentData;
                return new BufferedReader(new InputStreamReader(new ByteArrayInputStream(data), enc));
            }

            return super.getReader();
        }
    }

}
