package com.bxm.huola.message.common.constant;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;

import java.util.List;
import java.util.Set;

/**
 * 消息中心消息类型
 * 如有修改，请TM的同步至confluence3.3.11，否则后果自负
 *
 * @author liujia
 */
public enum MessageTypeEnum {
    /**
     * 互动消息
     */
    INTERACTION,
    /**
     * 系统消息
     */
    SYSTEM("系统通知", "https://m.wstong.com/localnews_prod/message/system.png"),
    /**
     * 分享被阅读消息
     */
    SHARE,
    /**
     * 评论消息
     */
    COMMENT("评论通知", "https://m.wstong.com/localnews_prod/message/comment.png"),
    /**
     * 点赞消息
     */
    LIKE("点赞通知", "https://m.wstong.com/localnews_prod/message/like.png"),
    /**
     * 邀请消息
     */
    INVITE,
    /**
     * 余额变动
     */
    BALANCE("余额变动", "https://m.wstong.com/localnews_prod/message/balance.png"),
    /**
     * 推送新闻
     */
    NEWS("今日热闻", "https://m.wstong.com/localnews_prod/message/news.png"),

    /**
     * 新增粉丝消息
     */
    ADD_FUNS("新增粉丝", "https://m.wstong.com/localnews_prod/material/icon_new_flower.png"),

    /**
     * 评论与点赞类型
     */
    LIKE_AND_COMMENT("评论点赞", "https://m.wstong.com/localnews_prod/material/icon_comment.png"),

    /**
     * 3.12.0新增 @我
     */
    AT_ME("@我的", "https://m.wstong.com/localnews_prod/material/icon_remind.png"),
    ;

    /**
     * 消息类目的文案
     */
    private String title;

    /**
     * 消息类目的图标
     */
    private String icon;

    MessageTypeEnum() {

    }

    MessageTypeEnum(String title, String icon) {
        this.title = title;
        this.icon = icon;
    }

    public String getTitle() {
        return title;
    }

    public String getIcon() {
        return icon;
    }

    /**
     * 消息列表中显示的类目
     *
     * @return 消息归类
     */
    public static List<MessageTypeEnum> getMessageCenterListType() {
        List<MessageTypeEnum> typeList = Lists.newArrayList();
        typeList.add(SYSTEM);
        typeList.add(NEWS);
        typeList.add(BALANCE);

        return typeList;
    }

    public static List<MessageTypeEnum> getTopType() {
        List<MessageTypeEnum> typeList = Lists.newArrayList();
        typeList.add(LIKE_AND_COMMENT);
        typeList.add(ADD_FUNS);
        typeList.add(AT_ME);

        return typeList;
    }

    /**
     * 消息类型是否为互动消息
     *
     * @param msgType 消息类型
     * @return true表示为互动消息
     */
    public static boolean isInteraction(String msgType) {
        return MessageTypeEnum.COMMENT.name().equals(msgType)
                || MessageTypeEnum.LIKE.name().equals(msgType)
                || MessageTypeEnum.INVITE.name().equals(msgType)
                || MessageTypeEnum.SHARE.name().equals(msgType)
                || MessageTypeEnum.AT_ME.name().equals(msgType);
    }

    /**
     * 互动消息的消息类型
     *
     * @return 消息列表
     */
    public static Set<MessageTypeEnum> getInteractionType() {
        return ImmutableSet.of(MessageTypeEnum.COMMENT,
                MessageTypeEnum.LIKE,
                MessageTypeEnum.INVITE,
                MessageTypeEnum.SHARE);
    }
}
