package com.bxm.thirdparty.payment.config;

import cn.hutool.core.collection.CollectionUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.thirdparty.payment.callback.PaymentCallbackService;
import com.bxm.thirdparty.payment.facade.PaymentFacadeService;
import com.bxm.thirdparty.payment.param.BindAccountRelation;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * 回调函数注册器
 *
 * @author liujia
 * @date 2022/07/01 09:24:33
 * @since 1.0.0
 */
@Slf4j
@Component
public class PaymentCallbackRegister implements ApplicationRunner {

    private static final Map<String, PaymentCallbackService> INSTANCE_MAP = Maps.newHashMap();

    @Value("${spring.application.name}")
    private String applicationName;

    /**
     * 获取订单类型对应的支付生命周期服务接口
     *
     * @param orderType 订单类型
     * @return 订单类型对应的生命周期处理
     */
    public PaymentCallbackService loadInstance(String orderType) {
        PaymentCallbackService paymentCallbackService = INSTANCE_MAP.get(orderType);
        if (null == paymentCallbackService) {
            log.error("请在配置文件中配置【payment.config.paymentTypeMap.{}】对应的服务", orderType);

            throw new IllegalArgumentException("未配置[" + orderType + "]对应的PaymentPhaseService");
        }

        return paymentCallbackService;
    }

    @SuppressWarnings("AlibabaThreadPoolCreation")
    @Override
    public void run(ApplicationArguments args) {
        Collection<PaymentCallbackService> phaseServices = SpringContextHolder.getBeans(PaymentCallbackService.class);

        Set<String> orderTypeSet = Sets.newHashSet();

        for (PaymentCallbackService phaseService : phaseServices) {
            log.info("注册订单类型[{}]对应的逻辑处理", phaseService.support());

            if (!orderTypeSet.add(phaseService.support())) {
                throw new IllegalArgumentException("[{}]定义重复，一个订单类型只能定义一个回调类");
            }

            INSTANCE_MAP.put(phaseService.support(), phaseService);
        }

        log.info("订单类型和回调绑定处理完成");

        ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.execute(() -> {
            log.info("开始执行绑定服务注册");
            try {
                this.createLazyRegisterTask();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            log.info("执行绑定服务注册完成");
        });
    }

    private void createLazyRegisterTask() {
        int retryTime = 5;
        while (true) {
            if (retryTime <= 0) {
                log.error("重试5次后，仍然注册支付失败，请检查配置是否正确或者支付服务在线");
                break;
            }
            try {
                TimeUnit.SECONDS.sleep(3);
            } catch (InterruptedException e) {
                log.error(e.getMessage(), e);
            }

            boolean allHit = true;

            for (Map.Entry<String, PaymentCallbackService> entry : INSTANCE_MAP.entrySet()) {
                log.info("订单类型[{}]进行绑定操作", entry.getKey());

                PaymentFacadeService paymentFacadeService = SpringContextHolder.getBean(PaymentFacadeService.class);

                PaymentCallbackService paymentCallbackService = entry.getValue();
                List<BindAccountRelation> bindAccountRelations = paymentCallbackService.bindRelation();
                if (CollectionUtil.isNotEmpty(bindAccountRelations)) {
                    Message message = paymentFacadeService.register(entry.getKey(), bindAccountRelations);
                    if (null == message || !message.isSuccess()) {
                        allHit = false;
                    } else {
                        log.info(
                            "订单支付服务注册成功，订单类型：{},场景绑定：{}",
                            entry.getKey(),
                            JSON.toJSONString(bindAccountRelations)
                        );
                    }
                }
            }

            if (allHit) {
                break;
            }
            retryTime--;
        }
        log.info("绑定订单账号服务完成");
    }
}
