package com.bxm.kylin.checker.checker.impl;

import com.bxm.kylin.checker.checker.CheckEnvironmentEnum;
import com.bxm.kylin.checker.checker.CheckResult;
import com.bxm.kylin.checker.checker.Checked;
import com.bxm.kylin.utils.DomainUtils;
import com.bxm.kylin.core.entity.CheckPlan;
import com.bxm.kylin.core.entity.Domain;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

/**
 * HTTP 检测器
 * @author allen
 * @date 2019/10/16
 * @since 1.0.0
 */
@Slf4j
@Configuration
public class HttpChecker extends AbstractChecker {

    private static final int DEFAULT_CONNECT_TIMEOUT = 500;
    private static final int DEFAULT_READ_TIMEOUT = 10000;

    @Override
    public CheckEnvironmentEnum getCheckEnvironment() {
        return CheckEnvironmentEnum.HTTP;
    }

    @Override
    public boolean isGlobalBlacklist() {
        return true;
    }

    @Override
    public Checked check(CheckPlan plan, Domain domain) {
        String wholeUrl = DomainUtils.getWholeUrl(domain);

        Checked checked = new Checked();
        checked.setCheckUrl(wholeUrl);
        checked.setCheckResult(check0(wholeUrl, plan, checked));
        return checked;
    }

    private CheckResult check0(String url, CheckPlan plan, Checked checked) {
        try {
            RestTemplate restTemplate = create(DEFAULT_CONNECT_TIMEOUT, DEFAULT_READ_TIMEOUT);
            ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
            int statusCodeValue = response.getStatusCodeValue();
            String body = response.getBody();
            // Setting checked object
            checked.setStatusCode(statusCodeValue);
            checked.setHeaders(JsonHelper.convert(response.getHeaders()));
            checked.setBody(body);

            Integer expectStatusCode = plan.getExpectStatusCode();
            String expectBodyRegex = plan.getExpectBodyRegex();
            boolean expectStatusCodeResult = false,
                    expectBodyRegexResult = false;
            if (null != expectStatusCode && statusCodeValue == expectStatusCode) {
                expectStatusCodeResult = true;
            }
            if (StringUtils.isBlank(expectBodyRegex)) {
                expectBodyRegexResult = true;
            }
            else if (StringUtils.isNotBlank(expectBodyRegex) && StringUtils.isNotBlank(body) && body.matches(expectBodyRegex)) {
                expectBodyRegexResult = true;
            }
            return expectStatusCodeResult && expectBodyRegexResult ? CheckResult.PASSED : CheckResult.NOPASS;
        } catch (RestClientException e) {
            if (log.isWarnEnabled()) {
                log.warn("check0: {} - {}", url, e.getMessage());
            }
            return CheckResult.NOPASS;
        }
    }

    private RestTemplate create(int connectTimeout, int readTimeout) {
        return new RestTemplateBuilder()
                .setConnectTimeout(connectTimeout)
                .setReadTimeout(readTimeout)
                .build();
    }
}
