package com.bxm.kylin.checker.job.scheduler;

import com.bxm.kylin.checker.checker.CheckEnvironmentEnum;
import com.bxm.kylin.checker.checker.Checker;
import com.bxm.kylin.checker.job.JobConstant;
import com.bxm.kylin.checker.job.MyJob;
import com.bxm.kylin.checker.job.worker.JobWorkerDispatcher;
import com.bxm.kylin.core.entity.CheckEnvironment;
import com.bxm.kylin.core.entity.CheckPlan;
import com.bxm.kylin.core.service.ICheckEnvironmentService;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.quartz.*;
import org.quartz.impl.StdSchedulerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Configuration;

import java.util.Date;
import java.util.Map;

/**
 * @author allen
 * @since 1.0.0
 */
@Configuration
public class QuartzJobScheduler implements JobScheduler, DisposableBean, ApplicationListener<ApplicationStartedEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(QuartzJobScheduler.class);

    private final ICheckEnvironmentService checkEnvironmentService;

    private final Map<Integer, Checker> checkerMapping = Maps.newHashMap();
    private Scheduler scheduler;
    private ConfigurableApplicationContext applicationContext;

    public QuartzJobScheduler(ICheckEnvironmentService checkEnvironmentService) {
        this.checkEnvironmentService = checkEnvironmentService;
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        this.applicationContext = event.getApplicationContext();
        this.postChecker(applicationContext);
        this.startScheduler(applicationContext);
    }

    private void startScheduler(ConfigurableApplicationContext applicationContext) {
        try {
            this.scheduler = StdSchedulerFactory.getDefaultScheduler();
            Map<String, JobListener> beans = applicationContext.getBeansOfType(JobListener.class);
            if (MapUtils.isNotEmpty(beans)) {
                for (JobListener listener : beans.values()) {
                    this.scheduler.getListenerManager().addJobListener(listener);
                }
            }
            this.scheduler.start();
        } catch (SchedulerException e) {
            throw new RuntimeException(e);
        }
    }

    private void postChecker(ConfigurableApplicationContext applicationContext) {
        Map<String, Checker> beans = applicationContext.getBeansOfType(Checker.class);
        for (Map.Entry<String, Checker> entry : beans.entrySet()) {
            Checker value = entry.getValue();
            CheckEnvironmentEnum checkEnvironment = value.getCheckEnvironment();
            checkerMapping.put(checkEnvironment.getType(), value);
        }
    }

    @Override
    public void destroy() {
        try {
            this.scheduler.shutdown(true);
        } catch (SchedulerException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("shutdown: ", e);
            }
        }
    }

    @Override
    public boolean trigger(CheckPlan checkPlan) {
        MyJob myJob = create(checkPlan);
        return triggerJob(myJob);
    }

    @Override
    public boolean remove(CheckPlan checkPlan) {
        return removeJob(new MyJob(checkPlan, applicationContext));
    }

    @Override
    public boolean triggerJob(MyJob job) {
        if (null == job) {
            return false;
        }

        JobDataMap jobDataMap = new JobDataMap();
        jobDataMap.put(JobConstant.JobDataMapField.JOB, job);
        JobDetail jobDetail = JobBuilder.newJob(JobWorkerDispatcher.class)
                .setJobData(jobDataMap)
                .withIdentity(job.getJobName(), job.getJobGroup())
                .build();
        CronTrigger trigger = TriggerBuilder.newTrigger()
                .withIdentity(job.getJobName(), job.getJobGroup())
                .withSchedule(CronScheduleBuilder.cronSchedule(job.getCronExpression()))
                .build();
        try {
            Date date = this.scheduler.scheduleJob(jobDetail, trigger);
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Job [{}-{}] registry successful, Next fire at: {}", job.getJobGroup(), job.getJobName(), date);
            }
            return true;
        } catch (SchedulerException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("", e);
            }
            return false;
        }
    }

    @Override
    public boolean removeJob(MyJob job) {
        try {
            boolean flag = this.scheduler.deleteJob(new JobKey(job.getJobName(), job.getJobGroup()));
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Job [{}-{}] has been removed!", job.getJobGroup(), job.getJobName());
            }
            return flag;
        } catch (SchedulerException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("", e);
            }
        }
        return false;
    }

    private MyJob create(CheckPlan checkPlan) {
        MyJob myJob = new MyJob(checkPlan, applicationContext);
        CheckEnvironment checkEnvironment = checkEnvironmentService.getById(checkPlan.getEnvironmentId());
        Checker checker = checkerMapping.get(checkEnvironment.getEnvironment());
        myJob.setChecker(checker);
        return myJob;
    }
}
