package com.bxm.kylin.checker.trigger;

import com.bxm.kylin.cache.CacheKey;
import com.bxm.kylin.checker.checker.CheckResult;
import com.bxm.kylin.checker.job.MyJob;
import com.bxm.kylin.core.entity.Domain;
import com.bxm.kylin.utils.DomainUtils;
import com.bxm.warcar.MessageException;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;

/**
 * 消息通知触发器。
 * @author allen
 * @date 2019/11/6
 * @since 1.0.0
 */
@Slf4j
@Configuration
public class NotifyTrigger extends AbstractTrigger {

    private static final int NOTIFY_MAXIMUM_TIMES = 3;
    private final Counter counter;
    private final Updater updater;
    private final MessageSender messageSender;

    public NotifyTrigger(Counter counter, Updater updater, MessageSender messageSender) {
        this.counter = counter;
        this.updater = updater;
        this.messageSender = messageSender;
    }

    @Override
    protected void handle(MyJob job, CheckResult result) {
    }

    @Override
    protected void doPassed(MyJob job) {
        Long checkPlanId = job.getId();
        if (getNotifyTimes(checkPlanId) == 0) {
            return;
        }
        String message = getMessage(job, false);
        this.sendMessage(message);
        this.setRecovered(checkPlanId);
    }

    @Override
    protected void doNoPass(MyJob job) {
        Long checkPlanId = job.getId();
        if (getNotifyTimes(checkPlanId) >= NOTIFY_MAXIMUM_TIMES) {
            return;
        }
        String message = getMessage(job, true);
        this.sendMessage(message);
        this.setWarning(checkPlanId);
    }

    private String getMessage(MyJob job, boolean warning) {
        StringBuilder str = new StringBuilder();

        if (warning) {
            str.append("【报警】");
        } else {
            str.append("【恢复】");
        }

        Domain domain = job.getDomain();

        str.append(DomainUtils.getWholeUrl(domain)).append(" ");
        str.append("在 ").append(job.getCheckEnvironment().getEnvironmentName()).append(" 环境下");

        if (warning) {
            str.append("监测不通过！");
        } else {
            str.append("已恢复。");
        }

        return str.toString();
    }

    private void sendMessage(String message) {
        try {
            messageSender.send2(new Message(message));
        } catch (MessageException e) {
            log.warn("sendMessage: {}", e.getMessage());
        }
    }

    private Long getNotifyTimes(Long checkPlanId) {
        return counter.hget(CacheKey.Trigger.warning(), String.valueOf(checkPlanId));
    }

    /**
     * 设置计划处于报警状态。
     * @param checkPlanId 计划ID
     */
    private void setWarning(Long checkPlanId) {
        counter.hincrementAndGet(CacheKey.Trigger.warning(), String.valueOf(checkPlanId));
    }

    /**
     * 设置计划处于恢复状态。
     * @param checkPlanId 计划ID
     */
    private void setRecovered(Long checkPlanId) {
        updater.hremove(CacheKey.Trigger.warning(), String.valueOf(checkPlanId));
    }
}
