package com.bxm.kylin._super.sdk;


import com.bxm.kylin._super.sdk.modal.*;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.bxm.warcar.message.dingding.DingDingMessageSender;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.web.util.UriComponentsBuilder;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author allen
 * @date 2021-08-23
 * @since 1.0
 */
@Slf4j
public class DefaultKylinImpl implements Kylin {

    private final KylinApiClient client;
    private final CacheKey cacheKey;
    private final Fetcher fetcher;
    private final Updater updater;
    private final MessageSender messageSender;
    private final boolean noticeIfNecessary;
    private final String webhook;
    private final ExecutorService wakeupThreadPool = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("wakeup"));

    public DefaultKylinImpl(KylinApiClient client, CacheKey cacheKey, Fetcher fetcher,
                            Updater updater, KylinProperties kylinProperties, String webhook) {
        this.client = client;
        this.cacheKey = cacheKey;
        this.fetcher = fetcher;
        this.updater = updater;
        this.messageSender = new DingDingMessageSender(kylinProperties.getDingTalkWebhookUrl());
        this.noticeIfNecessary = kylinProperties.isNoticeIfNecessary();
        this.webhook = webhook;
    }

    @Override
    public void refresh(Set<String> ids, String groupId) {
        Map<String, CheckPlan> id2Url = fetcher.hfetchall(cacheKey.hashCheckPlan(), CheckPlan.class);
        if (MapUtils.isNotEmpty(id2Url)) {
            for (Map.Entry<String, CheckPlan> entry : id2Url.entrySet()) {
                String id = entry.getKey();
                CheckPlan checkPlan = entry.getValue();
                if (CollectionUtils.isEmpty(ids) || ids.contains(id)) {
                    String url = getUrl(id);
                    this.close(id, false);
                    this.start(id, groupId, url, checkPlan.getRemark());
                }
            }
        }
    }

    @Override
    public void start(String id, String groupId, String url) {
        this.start(id, groupId, url, null);
    }

    @Override
    public void start(String id, String groupId, String url, String remark) {
        Map<String, Domain> domains = getDomains(id);
        Domain available = getFirstAvailableDomain(domains);
        if (Objects.isNull(available)) {
            List<Domain> disableList = filterCollect(domains, stringDomainEntry -> !stringDomainEntry.getValue().isAvailable());
            String path = UriComponentsBuilder.fromUriString(url).build().getPath();
            available = getFirstAvailableDomainOnKylin(id, groupId, disableList, path);
        }
        if (Objects.isNull(available)) {
            log.warn("No available domain of ticket '{}'", id);
            this.sendMessage(String.format("ID [%s] 没有微信环境可用的域名", id));
            return;
        }

        // 新的URL
        String newUrl = buildNewUrl(available, url);

        String oldUrl = getUrl(id);
        if (StringUtils.equals(oldUrl, newUrl)) {
            log.info("{} oldUrl: {}, newUrl: {}", id, oldUrl, newUrl);
            return;
        }

        updater.hupdate(cacheKey.hashAvailableUrl(), id, newUrl);
        updater.hupdate(cacheKey.hashDomains(id), TypeHelper.castToString(available.getId()), available);

        this.refreshCheckPlanIfNecessary(id, available, newUrl, remark);

        // 发送替换通知
        log.info("{} sourceUrl: {}, newUrl: {}", id, url, newUrl);
        if (noticeIfNecessary) {
            this.sendMessage(String.format("ID [%s] \n落地页链接\n %s \n当前已替换成\n %s", id, url, newUrl));
        }
    }

    @Override
    public void close(String id) {
        this.close(id, true);
    }

    @Override
    public void close(String id, boolean deleteHistoryDomains) {
        // 删除麒麟监测计划
        CheckPlan checkPlan = fetcher.hfetch(cacheKey.hashCheckPlan(), id, CheckPlan.class);
        if (Objects.nonNull(checkPlan)) {
            client.deleteCheckPlan(checkPlan.getId());
        }

        // 删除配置
        updater.hremove(cacheKey.hashAvailableUrl(), id);
        updater.hremove(cacheKey.hashCheckPlan(), id);
        if (deleteHistoryDomains) {
            updater.remove(cacheKey.hashDomains(id));
        }

        log.info("{} closed.", id);
    }

    @Override
    public void changed(CheckPlan checkPlan) {
        if (Objects.isNull(checkPlan)) {
            return;
        }
        boolean enable = checkPlan.isAvailable();

        Map<String, CheckPlan> map = fetcher.hfetchall(cacheKey.hashCheckPlan(), CheckPlan.class);
        if (MapUtils.isEmpty(map)) {
            return;
        }

        map.entrySet().forEach(new Consumer<Map.Entry<String, CheckPlan>>() {
            @Override
            public void accept(Map.Entry<String, CheckPlan> entry) {
                String id = entry.getKey();
                CheckPlan plan = entry.getValue();
                if (!Objects.equals(checkPlan.getId(), plan.getId())) {
                    return;
                }
                String domainId = TypeHelper.castToString(checkPlan.getDomainId());
                Domain domain = getDomain(id, domainId);
                if (Objects.isNull(domain)) {
                    return;
                }
                domain.setAvailable(enable);
                updater.hupdate(cacheKey.hashDomains(id), domainId, domain);

                log.info("ID {} domain {} has changed state: {}", id, domainId, enable);

                String groupId = domain.getGroupId();
                String url = getUrl(id);
                start(id, groupId, url, checkPlan.getRemark());
            }
        });
    }

    @Override
    public String getUrl(String id) {
        return fetcher.hfetch(cacheKey.hashAvailableUrl(), id, String.class);
    }

    @Override
    public Map<String, String> getAllUrl() {
        return fetcher.hfetchall(cacheKey.hashAvailableUrl(), String.class);
    }

    @Override
    public Map<String, String> getAllCheckPlan() {
        Map<String, CheckPlan> map = Optional.ofNullable(fetcher.hfetchall(cacheKey.hashCheckPlan(), CheckPlan.class)).orElse(new HashMap<>());
        Map<String, String> rst = Maps.newHashMap();
        map.forEach((key, value) -> rst.put(key, TypeHelper.castToString(value.getId())));
        return rst;
    }

    @Override
    public Map<String, CheckPlan> getAllCheckPlan2() {
        return Optional.ofNullable(fetcher.hfetchall(cacheKey.hashCheckPlan(), CheckPlan.class)).orElse(new HashMap<>());
    }

    @Override
    public CheckPlan getCheckPlan(String id) {
        return fetcher.hfetch(cacheKey.hashCheckPlan(),id, CheckPlan.class);
    }

    @Override
    public CheckPlan getSleepCheckPlan(String id) {
        return fetcher.hfetch(cacheKey.hashCheckPlanSleep(),id, CheckPlan.class);
    }

    @Override
    public void sleep(String id) {
        if (StringUtils.isBlank(id)) {
            return;
        }
        CheckPlan checkPlan = fetcher.hfetch(cacheKey.hashCheckPlan(), id, CheckPlan.class);
        if (Objects.isNull(checkPlan)) {
            throw new IllegalStateException("checkPlan");
        }
        checkPlan.setSleepTime(LocalDateTime.now());
        String url = getUrl(id);
        if (StringUtils.isBlank(url)) {
            throw new IllegalStateException("url");
        }
        checkPlan.setSleepUrl(url);
        Domain domain = getDomain(id, TypeHelper.castToString(checkPlan.getDomainId()));
        if (Objects.isNull(domain)) {
            throw new IllegalStateException("domain");
        }
        checkPlan.setSleepGroupId(domain.getGroupId());
        updater.hupdate(cacheKey.hashCheckPlanSleep(), id, checkPlan);
        log.info("[{}] is sleep", id);
        this.close(id, false);
    }

    @Override
    public void wakeup(String id) throws Exception {
        if (StringUtils.isBlank(id)) {
            return;
        }
        CheckPlan checkPlan = fetcher.hfetch(cacheKey.hashCheckPlanSleep(), id, CheckPlan.class);
        if (Objects.isNull(checkPlan)) {
            return;
        }
        CheckPlan exists = fetcher.hfetch(cacheKey.hashCheckPlan(), id, CheckPlan.class);
        if (Objects.nonNull(exists)) {
            log.info("[{}] has already waken up", id);
            updater.hremove(cacheKey.hashCheckPlanSleep(), id);
            return;
        }
        String groupId = checkPlan.getSleepGroupId();
        String url = checkPlan.getSleepUrl();
        String remark = checkPlan.getRemark();
        log.info("[{}] is wakeup", id);
        this.start(id, groupId, url, remark);
        // delete sleep value
        updater.hremove(cacheKey.hashCheckPlanSleep(), id);
    }

    @Override
    public void asyncWakeup(String id) {
        wakeupThreadPool.submit(() -> {
            try {
                wakeup(id);
            } catch (Exception e) {
                log.error("wakeup " + id, e);
            }
        });
    }

    @Override
    public CacheKey getCacheKey() {
        return this.cacheKey;
    }

    private List<Domain> filterCollect(Map<String, Domain> domains, Predicate<Map.Entry<String, Domain>> filter) {
        return domains.entrySet()
                .stream()
                .filter(filter)
                .map(Map.Entry::getValue).collect(Collectors.toList());
    }

    private Domain getFirstAvailableDomainOnKylin(String id, String groupId, List<Domain> exclude, String path) {
        List<Domain> domains = client.getAvailableDomains(groupId, Constants.UA_MP, Constants.IP, path);
        if (CollectionUtils.isEmpty(domains)) {
            log.warn("No available domain of group '{}'", groupId);
            this.sendMessage(String.format("域名组 [%s] 没有微信环境可用的域名", groupId));
            return null;
        }
        if (!CollectionUtils.isEmpty(exclude)) {
            Set<Long> excludeIds = exclude.stream().map(Domain::getId).collect(Collectors.toSet());
            domains.removeIf(domain -> excludeIds.contains(domain.getId()));
        }
        if (CollectionUtils.isEmpty(domains)) {
            log.warn("After remove, No available domain of group '{}'", groupId);
            this.sendMessage(String.format("ID [%s] 域名组 [%s] 没有微信环境可用的域名", id, groupId));
            return null;
        }
        int remain = domains.size() - 1;
        if (remain <= 2) {
            log.warn("id [{}] group [{}] available domain quantity less: {}", id, groupId, remain);
            this.sendMessage(String.format("ID [%s] 域名组 [%s] 微信环境剩余可用域名数为 %s", id, groupId, remain));
        }
        return domains.iterator().next();
    }

    /**
     * 获取当前广告券可用的域名id
     * @param domains 域名列表
     * @return domainId
     */
    private Domain getFirstAvailableDomain(Map<String, Domain> domains) {
        // 判断已保存的域名列表是否有可用的，如果有，那么直接结束了。
        if (MapUtils.isNotEmpty(domains)) {
            for (Map.Entry<String, Domain> entry : domains.entrySet()) {
                Domain domain = entry.getValue();
                boolean availableDomain = client.isAvailableDomain(domain.getId());
                if (domain.isAvailable()
                        // 远程的实时状态也必须是可用的。
                        && availableDomain) {
                    return domain;
                }
            }
        }
        return null;
    }

    private Map<String, Domain> getDomains(String id) {
        KeyGenerator hashDomains = cacheKey.hashDomains(id);
        return Optional.ofNullable(fetcher.hfetchall(hashDomains, Domain.class)).orElse(new HashMap<>());
    }

    private Domain getDomain(String id, String domainId) {
        return fetcher.hfetch(cacheKey.hashDomains(id), domainId, Domain.class);
    }

    /**
     * 根据 {@link Domain} 重新构建 {@code sourceUrl}，替换 schema / host 内容，生成一个新的链接并返回。
     * @param domain 域名对象
     * @param sourceUrl 原始链接
     * @return 新的链接。
     */
    private String buildNewUrl(Domain domain, String sourceUrl) {
        return UriComponentsBuilder.fromUriString(sourceUrl)
                .scheme(Protocol.of(domain.getProtocol()).getName())
                .host(domain.getDomain())
                .build()
                .toString();
    }

    /**
     * 如果有必要的话，刷新监测计划。
     *
     * @param id ID
     * @param domain 当前需要监测的域名
     * @param url 监测的URL
     * @param remark 备注
     */
    private void refreshCheckPlanIfNecessary(String id, Domain domain, String url, String remark) {
        CheckPlan current = fetcher.hfetch(cacheKey.hashCheckPlan(), id, CheckPlan.class);
        if (Objects.nonNull(current)) {
            // 如果当前计划是可用的，并且与当前域名以及监测路径一致则不需要更新。
            String path = UriComponentsBuilder.fromUriString(url).build().getPath();

            boolean alikeDomain = Objects.equals(current.getDomainId(), domain.getId());
            boolean alikePath = StringUtils.equals(current.getCheckPath(), path);

            if (current.isAvailable() && alikeDomain && alikePath) {
                return;
            }
        }

        CheckPlan latest = createCheckPlanWithRetry(id, domain, url, remark);

        if (Objects.isNull(latest)) {
            return; // 如果重试后仍然失败，则返回
        }

        updater.hupdate(cacheKey.hashCheckPlan(), id, latest);

        // 删除旧的计划
        if (Objects.nonNull(current)) {
            client.deleteCheckPlan(current.getId());
        }
    }

    private CheckPlan createCheckPlanWithRetry(String id, Domain domain, String url, String remark) {
        int maxRetries = 2;
        int retries = 0;
        CheckPlan latest = null;
        boolean isRetrySuccess = false;

        while (retries <= maxRetries) {
            CreateCheckPlanParam createCheckPlanParam = new CreateCheckPlanParam()
                    .setEnvId(Constants.ENV_ID)
                    .setDomainId(domain.getId())
                    .setSourceUrl(url)
                    .setRemark(remark)
                    .setWebhook(webhook);
            latest = client.createCheckPlan(createCheckPlanParam);

            if (Objects.nonNull(latest)) {
                isRetrySuccess = retries > 0; // 如果重试次数大于0，则表示是重试成功
                break; // 成功则退出循环
            } else {
                retries++;
                if (retries > maxRetries) {
                    this.sendMessage(String.format("ID [%s] 创建监测计划失败了！", id));
                } else {
                    this.sendMessage(String.format("ID [%s] 创建监测计划失败，正在进行第%d次重试...", id, retries));
                }
            }
        }

        if (isRetrySuccess) {
            this.sendMessage(String.format("ID [%s] 创建监测计划经过重试后成功！", id));
        }

        return latest;
    }

    private void sendMessage(String content) {
        try {
            this.messageSender.send2(new Message(content));
        } catch (Exception e) {
            log.warn("send2: ", e);
        }
    }
}
