package com.bxm.kylin._super.sdk.life;

import com.bxm.kylin._super.sdk.CacheKey;
import com.bxm.kylin._super.sdk.Kylin;
import com.bxm.warcar.cache.RedisLock;
import com.bxm.warcar.utils.LifeCycle;
import com.bxm.warcar.utils.UUIDHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;

import java.time.Duration;
import java.util.Map;
import java.util.Objects;

/**
 *
 * <pre>
 *     @Bean(initMethod = "init", destroyMethod = "destroy")
 *     public SleepScheduler sleepScheduler(Kylin shoppingKylin) {
 *         return new SleepScheduler(shoppingKylin);
 *     }
 * </pre>
 *
 * @author huxiao
 * @date 2023/9/12
 * @since 1.0.0
 */
@Slf4j
public class SleepScheduler extends LifeCycle {

    private final Kylin kylin;
    private final ThreadPoolTaskScheduler scheduler;
    private final String expression;

    private final RedisLock redisLock;

    /**
     * @param kylin      kylin instance
     * @param expression cron trigger expression
     * @param redisLock
     */
    public SleepScheduler(Kylin kylin, String expression, RedisLock redisLock) {
        this.kylin = kylin;
        this.expression = expression;
        this.redisLock = redisLock;
        this.scheduler = new ThreadPoolTaskScheduler();
    }

    @Override
    protected void doInit() {
        scheduler.setPoolSize(1);
        scheduler.initialize();

        scheduler.schedule(() -> {
            CacheKey cacheKey = kylin.getCacheKey();
            if (Objects.isNull(cacheKey)) {
                log.error("No cacheKey!");
                return;
            }
            String key = cacheKey.stringSleepSchedulerLock().generateKey();
            String rid = UUIDHelper.generate();
            boolean acquiredLock = false;
            try {
                acquiredLock = redisLock.tryLock(key, rid, (int) Duration.ofHours(1).getSeconds());
                if (acquiredLock) {
                    Map<String, String> checkPlanMap = kylin.getAllCheckPlan();
                    for (Map.Entry<String, String> entry : checkPlanMap.entrySet()) {
                        String id = entry.getKey();
                        kylin.sleep(id);
                    }
                }
            } catch (Exception e) {
                log.error("execute: ", e);
            } finally {
                if (acquiredLock) {
                    try {
                        redisLock.unLock(key, rid);
                    } catch (Exception e) {
                        log.error("unLock: ", e);
                    }
                }
            }
        }, triggerContext -> {
            CronTrigger cronTrigger = new CronTrigger(expression);
            return cronTrigger.nextExecutionTime(triggerContext);
        });
    }

    @Override
    protected void doDestroy() {
        this.scheduler.shutdown();
    }
}
