package com.bxm.lovelink.cms.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.dto.chat.ChatDto;
import com.bxm.lovelink.common.dal.entity.dto.chat.SessionMessageQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.chat.ChatSessionMessageVo;
import com.bxm.lovelink.common.dal.entity.vo.chat.ChatSessionVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserBasicInfoSimpleVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserBasicInfoVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeSimpleVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeVo;
import com.bxm.lovelink.common.dal.manager.ChatManager;
import com.bxm.lovelink.common.dal.mapping.UserInfoMapping;
import com.bxm.lovelink.common.dal.service.IUserService;
import lombok.Data;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Objects;

/**
 * 对话系统
 * @author tangxiao
 * @date 2025/6/26
 * @since 1.0
 */
@RestController
@RequestMapping("/chat")
public class ChatController {

    private final ChatManager chatManager;
    private final IUserService userService;

    public ChatController(ChatManager chatManager, IUserService userService) {
        this.chatManager = chatManager;
        this.userService = userService;
    }

    /**
     * 会话列表
     * @param request
     * @return
     */
    @GetMapping("/listSession")
    public ResponseEntity<List<ChatSessionVo>> listSession(@Validated ListSessionRequest request) {
        return ResponseEntity.ok(chatManager.listSession(Constants.Chat.STAFF_USER_ID, request.getChatContactRole()));
    }

    /**
     * 获取消息
     * @param queryDto
     * @return
     */
    @GetMapping("/listSessionMessage")
    public ResponseEntity<IPage<ChatSessionMessageVo>> listSessionMessage(@Validated SessionMessageQueryDto queryDto) {
        IPage<ChatSessionMessageVo> page = chatManager.listSessionMessage(queryDto);
        return ResponseEntity.ok(page);
    }

    /**
     * 发送消息(人工)
     * @return
     */
    @PostMapping("/sendMessage")
    public ResponseEntity<Void> sendMessage(@Validated @RequestBody ChatDto dto) {
        chatManager.sendMessage(dto);
        return ResponseEntity.ok().build();
    }

    /**
     * AI建议 流式回答
     * @return
     */
    @PostMapping("/aiStream")
    public ResponseEntity<StreamingResponseBody> aiStream(@Validated @RequestBody ChatDto chatDto) {
        StreamingResponseBody responseBody = outputStream -> {
            chatManager.aiStream(chatDto, outputStream);
        };
        HttpHeaders headers = new HttpHeaders();
        return ResponseEntity.ok().headers(headers).contentType(MediaType.TEXT_EVENT_STREAM).body(responseBody);
    }

    /**
     * 消除已读
     * @param request
     * @return
     */
    @PostMapping("/readMessage")
    public ResponseEntity<Void> readMessage(@Validated @RequestBody ReadMessageRequest request) {
        chatManager.readMessage(Constants.Chat.STAFF_USER_ID, request.getSessionId());
        return ResponseEntity.ok().build();
    }

    /**
     * 交还Ai托管
     * @param request
     * @return
     */
    @PostMapping("/switchChatRole")
    public ResponseEntity<Void> switchChatRole(@Validated @RequestBody SwitchChatRoleRequest request) {
        chatManager.switchChatRole(request.getSessionId(), request.getChatContactRole());
        return ResponseEntity.ok().build();
    }

    /**
     * 获取客户详情接口
     * @param userId
     * @return
     */
    @GetMapping("/getUserInfo")
    public ResponseEntity<Void> switchChatRole(@RequestParam("userId") Long userId) {
        UserCompositeVo vo = userService.getCompositeVoById(userId);
        if (vo == null) {
            return ResponseEntity.ok(null);
        }
        //设置值
        UserBasicInfoVo basicInfo = vo.getUserBasicInfo();
        UserCompositeSimpleVo simpleVo = UserInfoMapping.INSTANCE.voToVo(vo);
        UserBasicInfoSimpleVo simpleBasicInfo = simpleVo.getUserBasicInfo();
        if (Objects.equals(basicInfo.getSchoolHide(), Constants.NO)) {
            simpleBasicInfo.setSchoolName(null);
        }
        return ResponseEntity.ok().build();
    }

    @Data
    public static class ListSessionRequest {
        /**
         * tab：0 AI托管中、 1 人工
         */
        @NotNull(message = "chatContactRole不能为空")
        private Integer chatContactRole;
    }

    @Data
    public static class ReadMessageRequest {
        /**
         * 会话id
         */
        @NotNull(message = "sessionId不能为空")
        private Long sessionId;
    }

    @Data
    public static class SwitchChatRoleRequest {
        /**
         * 会话id
         */
        @NotNull(message = "sessionId不能为空")
        private Long sessionId;

        /**
         * 聊天：0 AI托管中、 1 人工
         */
        private Integer chatContactRole = Constants.Chat.CHAT_CONTACT_ROLE_AI;
    }

}
