package com.bxm.lovelink.cms.controller;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.bxm.lovelink.common.dal.entity.DecorationScoreDetail;
import com.bxm.lovelink.common.dal.entity.UserCms;
import com.bxm.lovelink.common.dal.entity.dto.user.UserCmsDto;
import com.bxm.lovelink.common.dal.entity.dto.user.UserQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeVo;
import com.bxm.lovelink.common.dal.service.IUserBasicInfoService;
import com.bxm.lovelink.common.dal.service.IUserCmsService;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Objects;

/**
 * 用户管理
 *
 * @author zhangdong
 * @date 2025/5/29
 */
@RestController
@RequestMapping("/user")
public class UserController {

    private final IUserBasicInfoService userBasicInfoService;
    private final IUserCmsService userCmsService;


    public UserController(IUserBasicInfoService userBasicInfoService, IUserCmsService userCmsService) {
        this.userBasicInfoService = userBasicInfoService;
        this.userCmsService = userCmsService;
    }

    /**
     * 查询用户列表
     *
     * @param dto 查询条件
     * @return 用户列表
     */
    @PostMapping("/list")
    public ResponseEntity<IPage<UserCompositeVo>> list(@RequestBody UserQueryDto dto) {
        return ResponseEntity.ok(userBasicInfoService.list(dto));
    }

    /**
     * 待审核数量
     *
     * @return 待审核数量
     */
    @GetMapping("/waitAuditCount")
    public ResponseEntity<Integer> waitAuditCount() {
        return ResponseEntity.ok(userCmsService.count(new LambdaUpdateWrapper<UserCms>()
                .eq(UserCms::getAuditStatus, UserCms.AUDIT_STATUS_WAIT)));
    }


    /**
     * 审核用户数据
     *
     * @param dto 审核数据
     * @return 空
     */
    @PostMapping("/audit")
    public ResponseEntity<Void> audit(@RequestBody @Validated UserCmsDto dto) {
        UserCms cms = userCmsService.getById(dto.getUserCmsId());
        if (cms == null) {
            throw new IllegalArgumentException("数据不存在");
        }
        if (Objects.equals(cms.getAuditStatus(), UserCms.AUDIT_STATUS_FINISH)) {
            throw new IllegalArgumentException("该用户已审核完成，无法继续审核");
        }
        int decorationScore = 0;
        for (DecorationScoreDetail detail : dto.getDecorationScoreDetail()) {
            decorationScore += detail.getScore();
        }
        if (decorationScore > 100) {
            throw new IllegalArgumentException("装修得分总和不能超过100分");
        }
        UserCms entity = new UserCms();
        entity.setComplianceStatus(dto.getComplianceStatus());
        entity.setComplianceNoReason(dto.getComplianceNoReason());
        entity.setAppearanceLevel(dto.getAppearanceLevel());
        entity.setDecorationScoreDetail(dto.getDecorationScoreDetail());
        entity.setDecorationScore(decorationScore);
        entity.setAuditStatus(UserCms.AUDIT_STATUS_FINISH);
        boolean update = userCmsService.update(entity, new LambdaUpdateWrapper<UserCms>()
                .eq(UserCms::getId, dto.getUserCmsId())
                .eq(UserCms::getAuditStatus, UserCms.AUDIT_STATUS_WAIT));
        if (!update) {
            throw new IllegalArgumentException("该用户已审核完成，无法继续审核");
        }
        return ResponseEntity.ok().build();
    }


}
