package com.bxm.lovelink.cms.controller.activity;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityEvaluation;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.ActivityUserSelection;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityCenterQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.activity.ActivityCenterDataVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.ActivityCenterVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.service.IActivityEvaluationService;
import com.bxm.lovelink.common.dal.service.IActivityRegistrationService;
import com.bxm.lovelink.common.dal.service.IActivityService;
import com.bxm.lovelink.common.dal.service.IActivityUserSelectionService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;


/**
 * 活动中心
 *
 * @author dll
 * @date 2025/6/24
 */
@Slf4j
@RestController
@RequestMapping("/activity/center")
public class ActivityCenterController {

    private final IActivityService activityService;
    private final IActivityEvaluationService activityEvaluationService;
    private final IActivityRegistrationService activityRegistrationService;
    private final IActivityUserSelectionService iActivityUserSelectionService;
    private final IActivityEvaluationService iActivityEvaluationService;

    private static final Integer ZERO = 0;

    public ActivityCenterController(IActivityService activityService, IActivityEvaluationService activityEvaluationService, IActivityRegistrationService activityRegistrationService, IActivityUserSelectionService iActivityUserSelectionService, IActivityEvaluationService iActivityEvaluationService) {
        this.activityService = activityService;
        this.activityEvaluationService = activityEvaluationService;
        this.activityRegistrationService = activityRegistrationService;
        this.iActivityUserSelectionService = iActivityUserSelectionService;
        this.iActivityEvaluationService = iActivityEvaluationService;
    }


    /**
     * 查询活动中心数据
     *
     * @return
     */
    @GetMapping("/getActivityCeterData")
    public ResponseEntity<ActivityCenterDataVo> getActivityCeterData() {
        ActivityCenterDataVo activityCenterDataVo = new ActivityCenterDataVo();
        // 在线活动: 当前正在进行中或待开始的活动总数。
        QueryWrapper onLineQueryWrapper = new QueryWrapper();
        onLineQueryWrapper.gt(Activity.ACTIVITY_END_TIME, LocalDateTime.now());
        activityCenterDataVo.setOnLineActivityCount(activityService.count(onLineQueryWrapper));
        //结束活动: 已经完成的活动总数。
        QueryWrapper endQueryWrapper = new QueryWrapper();
        endQueryWrapper.le(Activity.ACTIVITY_END_TIME, LocalDateTime.now());
        activityCenterDataVo.setEndActivityCount(activityService.count(endQueryWrapper));
        int totalEValuateCount = activityEvaluationService.count();
        // 活动评价: 所有活动收到的评价次数。
        activityCenterDataVo.setEvaluateActivityCount(totalEValuateCount);
        // 复参意愿: 所有评价中，用户选择“愿意再次参加”的比例。
        QueryWrapper repeatWishQueryWrapper = new QueryWrapper();
        repeatWishQueryWrapper.eq(ActivityEvaluation.RE_ATTEND, Constants.YES);
        int repeatWishCount = activityEvaluationService.count(repeatWishQueryWrapper);
        Double value = divideAndFormat(repeatWishCount, totalEValuateCount) * 100;
        activityCenterDataVo.setRepeatWish(value);
        //真实复参: 统计所有报名用户中，实际再次报名参加其他活动的用户比例。
        int totalCount = activityRegistrationService.count(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS));
        int realRepeatCount = activityRegistrationService.getBaseMapper().selectList(
                new QueryWrapper<ActivityRegistration>()
                        .select(ActivityRegistration.USER_ID)
                        .eq(ActivityRegistration.STATUS, Constants.ActivityRegistrationStatus.SUCCESS)
                        .groupBy(ActivityRegistration.USER_ID)
                        .having("COUNT(user_id) >= 2")
        ).size();
        activityCenterDataVo.setRealRepeat(divideAndFormat(realRepeatCount, totalCount) * 100);
        return ResponseEntity.ok(activityCenterDataVo);
    }


    /**
     * 列表查询
     *
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/list")
    public ResponseEntity<IPage<ActivityCenterVo>> list(ActivityCenterQueryDto dto) throws Exception {
        IPage<ActivityCenterVo> page = ((ActivityMapper) activityService.getBaseMapper()).selectByDto(new Page<>(dto.getCurrent(), dto.getSize()), dto);
        if (CollectionUtils.isEmpty(page.getRecords())) {
            return ResponseEntity.ok().build();
        }
        for (ActivityCenterVo activityCenterVo : page.getRecords()) {
            //活动报名人数
            QueryWrapper queryWrapper = new QueryWrapper();
            queryWrapper.eq(ActivityRegistration.ACTIVITY_ID, activityCenterVo.getActivityId());
            activityCenterVo.setRegistrationCount(activityRegistrationService.count(queryWrapper));
            //活动签到人数(用户选择记录表)
            QueryWrapper signInQueryWrapper = new QueryWrapper();
            signInQueryWrapper.eq(ActivityUserSelection.ACTIVITY_ID, activityCenterVo.getActivityId());
            activityCenterVo.setSignInCount(iActivityUserSelectionService.count(signInQueryWrapper));
            //平均得分（评价表）
            QueryWrapper evaluationQueryWrapper = new QueryWrapper();
            evaluationQueryWrapper.eq(ActivityUserSelection.ACTIVITY_ID, activityCenterVo.getActivityId());
            int count = iActivityEvaluationService.count(evaluationQueryWrapper);
            activityCenterVo.setEvaluationCount(count);
            evaluationQueryWrapper.select("sum(avg_score)");
            List<BigDecimal> sum = iActivityEvaluationService.getBaseMapper().selectObjs(evaluationQueryWrapper);
            Double value = 0.0;
            if (Objects.nonNull(sum.get(ZERO))) {
                value = sum.get(ZERO).doubleValue();
            }
            activityCenterVo.setAvgScore(divideAndFormat(value, count));
            //复参意愿（=该活动评论下，愿意复参的用户/ 该活动的评论总数）
            QueryWrapper repeatWishQueryWrapper = new QueryWrapper();
            repeatWishQueryWrapper.eq(ActivityEvaluation.ACTIVITY_ID, activityCenterVo.getActivityId());
            int totalEValuateCount = activityEvaluationService.count(repeatWishQueryWrapper);
            repeatWishQueryWrapper.eq(ActivityEvaluation.RE_ATTEND, Constants.YES);
            int repeatWishCount = activityEvaluationService.count(repeatWishQueryWrapper);
            activityCenterVo.setRepeatIndex(divideAndFormat(repeatWishCount, totalEValuateCount));
        }

        return ResponseEntity.ok(page);
    }


    public static Double divideAndFormat(double dividend, int divisor) {
        if (divisor == 0) {
            return dividend;
        }
        if (dividend == 0) {
            return 0.0;
        }
        BigDecimal result = BigDecimal.valueOf(dividend).divide(BigDecimal.valueOf(divisor), 2, RoundingMode.HALF_UP);
        return result.doubleValue();
    }
}
