package com.bxm.lovelink.cms.controller.activity;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityOrganizer;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.ActivityType;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityDto;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityGuestRecommendDto;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityQueryDto;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityUpdateDto;
import com.bxm.lovelink.common.dal.entity.validate.InsertGroup;
import com.bxm.lovelink.common.dal.entity.validate.UpdateGroup;
import com.bxm.lovelink.common.dal.entity.vo.activity.ActivityVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.mapping.activity.ActivityMapping;
import com.bxm.lovelink.common.dal.service.IActivityOrganizerService;
import com.bxm.lovelink.common.dal.service.IActivityRegistrationService;
import com.bxm.lovelink.common.dal.service.IActivityService;
import com.bxm.lovelink.common.dal.service.IActivityTypeService;
import com.bxm.lovelink.common.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;


/**
 * 活动管理
 *
 * @author dll
 * @date 2025/6/24
 */
@Slf4j
@RestController
@RequestMapping("/activity/manage")
public class ActivityController {

    private final IActivityService activityService;

    private final IActivityTypeService activityTypeService;
    private final IActivityOrganizerService organizerService;
    private final IActivityRegistrationService activityRegistrationService;

    public ActivityController(IActivityService activityService, IActivityTypeService activityTypeService, IActivityOrganizerService organizerService, IActivityRegistrationService activityRegistrationService) {
        this.activityService = activityService;
        this.activityTypeService = activityTypeService;
        this.organizerService = organizerService;
        this.activityRegistrationService = activityRegistrationService;
    }


    /**
     * 新增
     *
     * @param dto
     * @return
     */
    @PostMapping("/add")
    public ResponseEntity<Object> add(@RequestBody @Validated(InsertGroup.class) ActivityDto dto) {
        Activity activity = ActivityMapping.INSTANCE.dtoToDo(dto);
        if (dto.getDiscountLimitType() != null && dto.getDiscountLimitType() == Constants.DiscountLimitType.DATE) {
            activity.setDiscountStartTime(LocalDateTime.now());
        }
        activity.setActivityStatus(Constants.ActivityStatus.CLOSE);
        activityService.save(activity);
        return ResponseEntity.ok().build();
    }

    /**
     * 修改
     *
     * @param dto
     * @return
     */
    @PostMapping("/update")
    public ResponseEntity<Object> update(@RequestBody @Validated(UpdateGroup.class) ActivityDto dto) {
        Activity activity = activityService.getById(dto.getId());
        if (Objects.isNull(activity)) {
            return ResponseEntity.notFound().build();
        }
        if (Objects.nonNull(dto.getDiscountLimitType()) && Objects.equals(Constants.DiscountLimitType.COUNT, dto.getDiscountLimitType())) {
            if (activity.getDiscountRegisterCount() > dto.getDiscountLimitCount()) {
                throw new IllegalStateException("早鸟票报名人数为:" + activity.getDiscountRegisterCount() + ",已超过早鸟票限制名额");
            }
        }
        if (Objects.nonNull(dto.getMaleParticipantsLimit()) && dto.getMaleParticipantsLimit() < activity.getMaleRegisteredCount() ) {
            throw new IllegalStateException("男生报名人数为:" + activity.getMaleRegisteredCount() + ",已超过男生报名限制名额");
        }
        if (Objects.nonNull(dto.getFemaleParticipantsLimit()) && dto.getFemaleParticipantsLimit() < activity.getFemaleRegisteredCount() ) {
            throw new IllegalStateException("女生报名人数为:" + activity.getMaleRegisteredCount() + ",已超过女生报名限制名额");
        }
        if (Objects.nonNull(dto.getParticipantsLimit()) && dto.getParticipantsLimit() < activity.getFemaleRegisteredCount() + activity.getMaleRegisteredCount() ) {
            throw new IllegalStateException("报名人数为:" + activity.getMaleRegisteredCount() + activity.getMaleRegisteredCount() + ",已超过报名限制名额");
        }
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq(ActivityRegistration.ACTIVITY_ID, activity.getId());
        queryWrapper.eq(ActivityRegistration.STATUS, Constants.ActivityRegistrationStatus.WAIT_PAY);
        int count = activityRegistrationService.count(queryWrapper);
        if (count > 0) {
            if (!isEqual(dto.getRegisterPrice(), activity.getRegisterPrice())
                    || !isEqual(dto.getDiscountPrice(), activity.getDiscountPrice())) {
                throw new IllegalStateException("存在待支付订单，不能修改价格");
            }
        }


        activityService.updateById(ActivityMapping.INSTANCE.dtoToDo(dto));
        return ResponseEntity.ok().build();
    }
    public boolean isEqual(BigDecimal a, BigDecimal b) {
        if (a == null && b == null) {
            return true;
        }
        if (a == null || b == null) {
            return false;
        }
        return a.compareTo(b) == 0;
    }
    /**
     * 修改嘉宾推荐
     *
     * @param dto
     * @return
     */
    @PostMapping("/updateGuestRecommend")
    public ResponseEntity<Object> updateGuestRecommend(@RequestBody @Validated(UpdateGroup.class) ActivityGuestRecommendDto dto) {
        ((ActivityMapper) activityService.getBaseMapper()).updateGuestRecommend(dto.getId(), dto.getGuestRecommend());
        return ResponseEntity.ok().build();
    }

    /**
     * 根据活动ID，查找活动详情
     *
     * @param id
     * @return
     */
    @GetMapping("/findById")
    public ResponseEntity<ActivityVo> findById(@RequestParam("id") Long id) {
        ActivityVo activityVo = ActivityMapping.INSTANCE.actToActVo(activityService.getById(id));
        return ResponseEntity.ok(activityVo);
    }


    /**
     * 列表查询
     *
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/list")
    public ResponseEntity<IPage<ActivityVo>> list(ActivityQueryDto dto) throws Exception {
        List<Integer> activityTypeIdsList = new ArrayList<>();
        if (Objects.nonNull(dto.getType())) {
            QueryWrapper queryWrapper = new QueryWrapper();
            queryWrapper.eq(ActivityType.TYPE, dto.getType());
            List<ActivityType> result = activityTypeService.list(queryWrapper);
            if (CollectionUtils.isEmpty(result)) {
                return ResponseEntity.ok().build();
            }
            result.stream().map(ActivityType::getId).forEach(activityTypeIdsList::add);
        }
        LambdaQueryWrapper<Activity> queryWrapper = new LambdaQueryWrapper<Activity>()
                .like(StringUtils.isNotEmpty(dto.getActivityName()), Activity::getName, dto.getActivityName())
                .eq(StringUtils.isNotEmpty(dto.getCityCode()), Activity::getCityCode, dto.getCityCode())
                .eq(Objects.nonNull(dto.getOvertStatus()), Activity::getOvertStatus, dto.getOvertStatus())
                .eq(Objects.nonNull(dto.getActivityStatus()), Activity::getActivityStatus, dto.getActivityStatus())
                .in(CollectionUtils.isNotEmpty(activityTypeIdsList), Activity::getActivityTypeId, activityTypeIdsList)
                .orderByDesc(Activity::getActivityStatus,Activity::getCreateTime);

        IPage<ActivityVo> voPage = ActivityMapping.INSTANCE.toActVoPage(activityService.getBaseMapper()
                .selectPage(new Page<>(dto.getCurrent(), dto.getSize()), queryWrapper));

        for (ActivityVo activityVo : voPage.getRecords()) {
            ActivityType activityType = activityTypeService.getById(activityVo.getActivityTypeId());
            activityVo.setActivityTypeName(Objects.nonNull(activityType) ? activityType.getName() : null);
            ActivityOrganizer activityOrganizer = organizerService.getById(activityVo.getOrganizerId());
            activityVo.setOrganizerName(Objects.nonNull(activityOrganizer) ? activityOrganizer.getName() : null);
        }
        return ResponseEntity.ok(voPage);
    }

    /**
     * 删除
     *
     * @param id
     * @return
     */
    @PostMapping("/delete")
    public ResponseEntity<Object> delete(@RequestParam("id") Long id) {
        Activity activity = activityService.getById(id);
        if (Objects.isNull(activity)) {
            throw new BusinessException("删除的数据不存在");
        }
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq(ActivityRegistration.ACTIVITY_ID, id);
        int count = activityRegistrationService.count(queryWrapper);
        if (count > 0) {
            throw new BusinessException("有报名人数的活动不能删除");
        }
        activityService.removeById(id);
        return ResponseEntity.ok().build();
    }


    /**
     * 修改活动状态
     *
     * @param dto
     * @return
     */
    @PostMapping("/updateStatus")
    public ResponseEntity<Object> updateStatus(@RequestBody @Validated(UpdateGroup.class) ActivityUpdateDto dto) {
        Activity activity = activityService.getById(dto.getId());
        if (Objects.isNull(activity)) {
            throw new BusinessException("修改的数据不存在");
        }
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq(ActivityRegistration.ACTIVITY_ID, dto.getId());
        int count = activityRegistrationService.count(queryWrapper);
        if (count > 0) {
            throw new BusinessException("有报名人数的活动不能修改状态");
        }
        activity.setActivityStatus(dto.getActivityStatus());
        activityService.updateById(activity);
        return ResponseEntity.ok().build();
    }
}
