package com.bxm.lovelink.cms.controller.activity;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.UserOrder;
import com.bxm.lovelink.common.dal.entity.dto.activity.registration.ActivityRegistrationQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.activity.registration.ActivityRegistrationVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.registration.ActivityUserRegistrationVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.mapper.ActivityRegistrationMapper;
import com.bxm.lovelink.common.dal.service.IActivityRegistrationService;
import com.bxm.lovelink.common.dal.service.IUserOrderService;
import com.bxm.lovelink.common.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Objects;


/**
 * 活动中心-报名表
 *
 * @author dll
 * @date 2025/6/24
 */
@Slf4j
@RestController
@RequestMapping("/activity/registration")
public class ActivityRegistrationController {

    private final IActivityRegistrationService iActivityRegistrationService;

    private final IUserOrderService userOrderService;
    private final ActivityMapper activityMapper;

    public ActivityRegistrationController(IActivityRegistrationService iActivityRegistrationService, IUserOrderService userOrderService, ActivityMapper activityMapper) {
        this.iActivityRegistrationService = iActivityRegistrationService;
        this.userOrderService = userOrderService;
        this.activityMapper = activityMapper;
    }


    /**
     * 退款
     *
     * @param id
     * @return
     */
    @Transactional
    @PostMapping("/refund")
    public ResponseEntity<Object> update(@RequestParam("id") Long id) {
        ActivityRegistration activityRegistration = iActivityRegistrationService.getById(id);
        if (Objects.isNull(activityRegistration)) {
            throw new BusinessException("要退款的数据不存在");
        }
        if (StringUtils.isBlank(activityRegistration.getOrderNo())) {
            throw new BusinessException("免费订单不能退款");
        }
        if (activityRegistration.getStatus() != Constants.ActivityRegistrationStatus.SUCCESS) {
            throw new BusinessException("不是支付成功的订单不能退款");
        }
        activityRegistration.setStatus(Constants.ActivityRegistrationStatus.REFUND);
        iActivityRegistrationService.updateById(activityRegistration);
        //开始处理退款后续操作
        //如果是享用早鸟票优惠，则 活动总人数，与早鸟票人数都减1
        Long activityId = activityRegistration.getActivityId();
        if (Objects.equals(Constants.YES, activityRegistration.getDiscountApplied())) {
            activityMapper.releaseDiscountRegisterCount(activityId);
        }
        if (Constants.UserGender.MALE == activityRegistration.getGender()) {
            activityMapper.releaseMaleRegisterCount(activityId);
        } else if (Constants.UserGender.FEMALE == activityRegistration.getGender()) {
            activityMapper.releaseFemaleRegisterCount(activityId);
        }
        //开始处理订单状态
        if (StringUtils.isBlank(activityRegistration.getOrderNo())) {
            return ResponseEntity.ok().build();
        }
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq(UserOrder.ORDER_NUM, activityRegistration.getOrderNo());
        UserOrder userOrder = userOrderService.getOne(queryWrapper);
        userOrder.setStatus(Constants.UserOrder.STATUS_REFUND_SUCCESS);
        userOrderService.updateById(userOrder);
        return ResponseEntity.ok().build();
    }


    /**
     * 列表查询
     *
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/list")
    public ResponseEntity<IPage<ActivityRegistrationVo>> list(ActivityRegistrationQueryDto dto) throws Exception {
        IPage<ActivityRegistrationVo> page = ((ActivityRegistrationMapper) iActivityRegistrationService.getBaseMapper()).selectByDto(new Page<>(dto.getCurrent(), dto.getSize()), dto);
        return ResponseEntity.ok(page);
    }


    /**
     * 根据性别查询列表
     *
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/guestRecommendList")
    public ResponseEntity<List<ActivityUserRegistrationVo>> guestRecommendList(ActivityRegistrationQueryDto dto) {
        if (Objects.isNull(dto.getActivityId()) || Objects.isNull(dto.getGender())) {
            throw new BusinessException("活动ID或者性别不能为空");
        }
        List<ActivityUserRegistrationVo> result = ((ActivityRegistrationMapper) iActivityRegistrationService.getBaseMapper())
                .guestRecommendList(dto);
        return ResponseEntity.ok(result);
    }

}
