package com.bxm.lovelink.cms.exception;

import com.bxm.lovelink.common.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.List;
import java.util.Set;

/**
 * @author huxiao
 * @date 2024/3/15
 * @since 1.0.0
 */
@ControllerAdvice
@Slf4j
public class GlobalExceptionHandler {

    @ExceptionHandler({BindException.class})
    public ResponseEntity<String> bindException(BindException throwable) {
        StringBuilder msg = new StringBuilder();
        List<ObjectError> errors = throwable.getAllErrors();
        for (ObjectError error : errors) {
            msg.append(error.getDefaultMessage()).append(",");
        }
        msg.deleteCharAt(msg.length() - 1);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(msg.toString());
    }

    @ExceptionHandler({MethodArgumentNotValidException.class})
    public ResponseEntity<String> methodArgumentNotValidException(MethodArgumentNotValidException throwable) {
        StringBuilder msg = new StringBuilder();
        BindingResult bindingResult = throwable.getBindingResult();
        List<ObjectError> errors = bindingResult.getAllErrors();
        for (ObjectError error : errors) {
            msg.append(error.getDefaultMessage()).append(",");
        }
        msg.deleteCharAt(msg.length() - 1);
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(msg.toString());
    }


    @ExceptionHandler(ConstraintViolationException.class)
    public ResponseEntity<String> constraintViolationException(ConstraintViolationException throwable) {
        StringBuilder msg = new StringBuilder();
        Set<ConstraintViolation<?>> constraintViolations = throwable.getConstraintViolations();
        for (ConstraintViolation<?> constraintViolation : constraintViolations) {
            msg.append(constraintViolation.getMessage());
        }
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(msg.toString());
    }

    @ExceptionHandler({BusinessException.class})
    public ResponseEntity<String> businessException(BusinessException throwable) {
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(throwable.getMessage());
    }

    @ExceptionHandler({HttpMessageNotReadableException.class
            , IllegalArgumentException.class
            , HttpRequestMethodNotSupportedException.class
    })
    public ResponseEntity<String> clientException(Throwable throwable) {
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(throwable.getMessage());
    }

    @ExceptionHandler({Exception.class})
    public ResponseEntity<String> exception(Exception throwable) {
        log.error("system error", throwable);
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(throwable.getMessage());
    }
}
