package com.bxm.lovelink.common.cache;

import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.xcache.Target;
import com.bxm.warcar.xcache.TargetFactory;
import com.bxm.warcar.xcache.fetchers.LoadingCacheFetcher;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.*;

/**
 * @author zhangdong
 * @date 2025/7/2
 */
public abstract class AbstractTableCache<T> {

    private final Fetcher fetcher;
    private final Updater updater;
    private final RedissonClient redissonClient;
    private final LoadingCacheFetcher loadingCacheFetcher;
    private final ThreadPoolExecutor executor;

    public AbstractTableCache(Fetcher fetcher, Updater updater, RedissonClient redissonClient, LoadingCacheFetcher loadingCacheFetcher) {
        this.fetcher = fetcher;
        this.updater = updater;
        this.redissonClient = redissonClient;
        this.loadingCacheFetcher = loadingCacheFetcher;
        this.executor = new ThreadPoolExecutor(50, 100, 60, TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(1000),
                new NamedThreadFactory("cache-executor"));
    }


    public abstract KeyGenerator getKeyGenerator(Long userId);

    public abstract T getFromDatabase(Long userId);

    public abstract Class<T> getCacheClass();

    /**
     * @return 是否开启更新锁
     */
    protected boolean isLockingForUpdate() {
        return true;
    }

    /**
     * @return 返回缓存过期时间，单位秒。小于等于0表示不过期
     */
    protected int getExpireTime() {
        return 600;
    }


    public T getFromCache(Long userId) {
        return fetcher.fetch(getKeyGenerator(userId), getCacheClass());
    }

    public T getFromCache(Long userId, DataExtractor<T> dataExtractor) {
        return fetcher.fetch(getKeyGenerator(userId), dataExtractor, getCacheClass(), getExpireTime());
    }

    public void removeCache(Long userId) {
        if (Objects.isNull(userId)) {
            return;
        }
        updater.remove(getKeyGenerator(userId));
    }

    /**
     * 删除缓存后，从数据库获取数据并放入缓存
     * @param userId ID
     */
    public void preparationAfterRemoved(Long userId) {
        if (Objects.isNull(userId)) {
            return;
        }
        removeCache(userId);
        getIfNullFromDatabase(userId);
    }


    public List<T> getBatchFromAuto(List<Long> userIds) {
        List<T> list = Lists.newArrayList();
        if (CollectionUtils.isEmpty(userIds)) {
            return list;
        }
        if (userIds.size() == 1) {
            // 如果只有一个，直接查询，避免线程切换开销
            return Lists.newArrayList(getFromAuto(userIds.get(0)));
        }
        CountDownLatch countDownLatch = new CountDownLatch(userIds.size());
        List<Future<T>> futures = Lists.newArrayList();
        for (Long userId : userIds) {
            futures.add(executor.submit(() -> {
                try {
                    return getFromAuto(userId);
                } finally {
                    countDownLatch.countDown();
                }
            }));
        }
        try {
            countDownLatch.await();
        } catch (Exception e) {
            throw new BusinessException("cache wait error");
        }
        boolean hasError = false;
        for (Future<T> future : futures) {
            if (future.isDone()) {
                try {
                    T t = future.get();
                    if (t != null) {
                        list.add(future.get());
                    }
                } catch (Exception e) {
                    hasError = true;
                }
            } else {
                hasError = true;
            }
        }
        if (hasError) {
            throw new BusinessException("cache error");
        }
        return list;
    }

    public T getFromAuto(Long userId) {
        Integer isFromCache = Constants.YES;
        try {
            Target<Integer> target = new TargetFactory<Integer>()
                    .keyGenerator(RedisKeys.cacheMark())
                    .cls(Integer.class)
                    .build();
            isFromCache = loadingCacheFetcher.fetch(target);
        } catch (Exception ignored) {
        }
        if (Objects.equals(isFromCache, Constants.YES)) {
            return getIfNullFromDatabase(userId);
        }
        return getFromDatabase(userId);
    }


    public T getIfNullFromDatabase(Long userId) {
        if (isLockingForUpdate()) {
            return getIfNullFromDatabaseOnLock(userId);
        } else {
            return getIfNullFromDatabaseOutLock(userId);
        }
    }

    private T getIfNullFromDatabaseOutLock(Long userId) {
        return getFromCache(userId, () -> getFromDatabase(userId));
    }

    private T getIfNullFromDatabaseOnLock(Long userId) {
        T t = getFromCache(userId);
        if (t != null) {
            return t;
        }
        String key = userId + ":" + this.getClass().getSimpleName();
        RLock lock = redissonClient.getLock(RedisKeys.cacheSyncLock(key).generateKey());
        boolean b = false;
        try {
            b = lock.tryLock();
        } catch (Exception ignored) {
        }
        if (!b) {
            return getFromDatabase(userId);
        }
        try {
            return getFromCache(userId, () -> getFromDatabase(userId));
        } finally {
            lock.unlock();
        }
    }
}
